import { useState } from "react";
import { Code2, Wallet, CreditCard, Bot, Package, FileCode } from "lucide-react";
import { Button } from "@/components/ui/button";
import { cn } from "@/lib/utils";
import { EnhancedFileTree } from "@/components/studio/EnhancedFileTree";
import { CodeViewer } from "@/components/studio/CodeViewer";
import { ModuleBuilder } from "@/components/studio/ModuleBuilder";
import { SmartWallet } from "@/components/studio/SmartWallet";
import { SubscriptionPlans } from "@/components/studio/SubscriptionPlans";
import { AICodeAssistant } from "@/components/studio/AICodeAssistant";

type ViewType = "code" | "modules" | "wallet" | "subscription" | "ai-assistant";

const SmartFriendStudio = () => {
  const [activeView, setActiveView] = useState<ViewType>("code");
  const [selectedFile, setSelectedFile] = useState<string | null>(null);

  const menuItems = [
    { id: "code" as ViewType, label: "Code Editor", icon: Code2 },
    { id: "modules" as ViewType, label: "Module Builder", icon: Package },
    { id: "ai-assistant" as ViewType, label: "AI Assistant", icon: Bot },
    { id: "wallet" as ViewType, label: "Smart Wallet", icon: Wallet },
    { id: "subscription" as ViewType, label: "Subscriptions", icon: CreditCard },
  ];

  const renderContent = () => {
    switch (activeView) {
      case "code":
        return (
          <div className="flex h-full">
            <div className="w-64">
              <EnhancedFileTree onFileSelect={setSelectedFile} selectedFile={selectedFile} />
            </div>
            <div className="flex-1">
              <CodeViewer selectedFile={selectedFile} />
            </div>
          </div>
        );
      case "modules":
        return <ModuleBuilder />;
      case "wallet":
        return <SmartWallet />;
      case "subscription":
        return <SubscriptionPlans />;
      case "ai-assistant":
        return <AICodeAssistant />;
      default:
        return null;
    }
  };

  return (
    <div className="flex h-screen bg-background">
      <aside className="w-64 bg-card border-r border-border flex flex-col">
        <div className="p-6 border-b border-border">
          <div className="flex items-center gap-2">
            <FileCode className="h-6 w-6 text-primary" />
            <div>
              <h2 className="text-lg font-bold text-foreground">Smart Friend</h2>
              <p className="text-xs text-muted-foreground">Developer Studio 2.0</p>
            </div>
          </div>
        </div>

        <nav className="flex-1 p-4 space-y-2">
          {menuItems.map((item) => {
            const Icon = item.icon;
            return (
              <Button
                key={item.id}
                onClick={() => setActiveView(item.id)}
                variant={activeView === item.id ? "default" : "ghost"}
                className={cn(
                  "w-full justify-start gap-3",
                  activeView === item.id && "bg-primary text-primary-foreground"
                )}
              >
                <Icon className="h-5 w-5" />
                <span>{item.label}</span>
              </Button>
            );
          })}
        </nav>

        <div className="p-4 border-t border-border">
          <div className="text-xs text-muted-foreground text-center">
            <p>Smart Friend Studio v2.0</p>
            <p className="mt-1">© 2025 All rights reserved</p>
          </div>
        </div>
      </aside>

      <main className="flex-1 overflow-hidden">
        {renderContent()}
      </main>
    </div>
  );
};

export default SmartFriendStudio;
