import { useState } from "react";
import { WalletGuard } from "@/components/ide/WalletGuard";
import { Card } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { TrendingUp, Calendar, BarChart3, Sparkles } from "lucide-react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

const SmartMarketer = () => {
  const { toast } = useToast();
  const [title, setTitle] = useState("");
  const [content, setContent] = useState("");
  const [platform, setPlatform] = useState("facebook");
  const [generating, setGenerating] = useState(false);

  const generateContent = async () => {
    setGenerating(true);
    try {
      const { data, error } = await supabase.functions.invoke("ai-code-builder", {
        body: {
          operation: "generate",
          context: `Generate marketing content for ${platform}: ${title}`,
        },
      });
      if (error) throw error;
      setContent(data?.result || "");
      toast({ title: "Content generated successfully" });
    } catch (error: any) {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    } finally {
      setGenerating(false);
    }
  };

  const saveCampaign = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { error } = await supabase.from("marketing_campaigns").insert({
        user_id: user.id,
        title,
        content,
        platform,
        status: "draft",
      });

      if (error) throw error;
      toast({ title: "Campaign saved" });
      setTitle("");
      setContent("");
    } catch (error: any) {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    }
  };

  return (
    <WalletGuard>
      <div className="min-h-screen bg-background p-6">
        <div className="max-w-7xl mx-auto">
          <div className="flex items-center gap-3 mb-6">
            <TrendingUp className="h-8 w-8 text-primary" />
            <div>
              <h1 className="text-3xl font-bold text-foreground">Smart Marketer</h1>
              <p className="text-muted-foreground">AI-powered marketing content generator</p>
            </div>
          </div>

          <Tabs defaultValue="create" className="space-y-6">
            <TabsList>
              <TabsTrigger value="create">Create Campaign</TabsTrigger>
              <TabsTrigger value="campaigns">My Campaigns</TabsTrigger>
              <TabsTrigger value="analytics">Analytics</TabsTrigger>
            </TabsList>

            <TabsContent value="create">
              <Card className="p-6">
                <div className="space-y-4">
                  <div>
                    <label className="text-sm font-medium mb-2 block">Campaign Title</label>
                    <Input
                      value={title}
                      onChange={(e) => setTitle(e.target.value)}
                      placeholder="Enter campaign title"
                    />
                  </div>

                  <div>
                    <label className="text-sm font-medium mb-2 block">Platform</label>
                    <Select value={platform} onValueChange={setPlatform}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="facebook">Facebook</SelectItem>
                        <SelectItem value="google">Google Ads</SelectItem>
                        <SelectItem value="youtube">YouTube</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <label className="text-sm font-medium mb-2 block">Content</label>
                    <Textarea
                      value={content}
                      onChange={(e) => setContent(e.target.value)}
                      placeholder="Your marketing content..."
                      rows={8}
                    />
                  </div>

                  <div className="flex gap-2">
                    <Button onClick={generateContent} disabled={generating || !title}>
                      <Sparkles className="h-4 w-4 mr-2" />
                      {generating ? "Generating..." : "Generate with AI"}
                    </Button>
                    <Button onClick={saveCampaign} variant="outline" disabled={!title || !content}>
                      Save Campaign
                    </Button>
                  </div>
                </div>
              </Card>
            </TabsContent>

            <TabsContent value="campaigns">
              <Card className="p-6">
                <p className="text-muted-foreground">Your campaigns will appear here</p>
              </Card>
            </TabsContent>

            <TabsContent value="analytics">
              <Card className="p-6">
                <div className="flex items-center gap-2 mb-4">
                  <BarChart3 className="h-5 w-5 text-primary" />
                  <h3 className="text-lg font-semibold">Campaign Analytics</h3>
                </div>
                <p className="text-muted-foreground">Analytics dashboard coming soon</p>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </WalletGuard>
  );
};

export default SmartMarketer;
