import { useState, useEffect, useMemo } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { toast } from "sonner";
import { supabase } from "@/integrations/supabase/client";
import { Wallet, Send, History, DollarSign, ArrowUpRight, ArrowDownRight, RefreshCw, Search, Filter, X, Info, Tag, Plus } from "lucide-react";
import { z } from "zod";

const sendMoneySchema = z.object({
  recipientPhone: z.string().min(10, "Phone number must be at least 10 digits").max(15, "Phone number too long"),
  amount: z.number().positive("Amount must be positive").max(1000000, "Amount too large"),
  note: z.string().max(200, "Note too long").optional()
});

const requestMoneySchema = z.object({
  recipientPhone: z.string().min(10, "Phone number must be at least 10 digits").max(15, "Phone number too long"),
  amount: z.number().positive("Amount must be positive").max(1000000, "Amount too large"),
  note: z.string().max(200, "Note too long").optional()
});

type Transaction = {
  id: string;
  sender_id: string;
  recipient_id: string;
  amount: number;
  type: string;
  status: string;
  note: string | null;
  tags: string[] | null;
  created_at: string;
  updated_at: string;
  sender_profile?: { full_name: string; phone: string };
  recipient_profile?: { full_name: string; phone: string };
};

type Wallet = {
  id: string;
  user_id: string;
  balance: number;
  currency: string;
  updated_at: string;
};

type PaymentRequest = {
  id: string;
  requester_id: string;
  recipient_id: string;
  amount: number;
  note: string | null;
  status: string;
  created_at: string;
  requester_profile?: { full_name: string; phone: string };
  recipient_profile?: { full_name: string; phone: string };
};

const SmartPay = () => {
  const [wallet, setWallet] = useState<Wallet | null>(null);
  const [transactions, setTransactions] = useState<Transaction[]>([]);
  const [paymentRequests, setPaymentRequests] = useState<PaymentRequest[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isSending, setIsSending] = useState(false);
  const [recipientPhone, setRecipientPhone] = useState("");
  const [amount, setAmount] = useState("");
  const [note, setNote] = useState("");
  const [requestRecipientPhone, setRequestRecipientPhone] = useState("");
  const [requestAmount, setRequestAmount] = useState("");
  const [requestNote, setRequestNote] = useState("");
  const [isRequesting, setIsRequesting] = useState(false);
  
  // Filter states
  const [searchQuery, setSearchQuery] = useState("");
  const [filterType, setFilterType] = useState<string>("all");
  const [filterStatus, setFilterStatus] = useState<string>("all");
  const [minAmount, setMinAmount] = useState("");
  const [maxAmount, setMaxAmount] = useState("");
  const [startDate, setStartDate] = useState("");
  const [endDate, setEndDate] = useState("");
  const [showFilters, setShowFilters] = useState(false);
  
  // Transaction details modal states
  const [selectedTransaction, setSelectedTransaction] = useState<Transaction | null>(null);
  const [isDetailsModalOpen, setIsDetailsModalOpen] = useState(false);
  const [newTag, setNewTag] = useState("");
  const [isAddingTag, setIsAddingTag] = useState(false);

  useEffect(() => {
    fetchWallet();
    fetchTransactions();
    fetchPaymentRequests();

    // Subscribe to wallet and payment request changes for real-time updates
    const setupRealtimeSubscription = async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const channel = supabase
        .channel('smart-pay-realtime')
        .on(
          'postgres_changes',
          {
            event: 'UPDATE',
            schema: 'public',
            table: 'wallets',
            filter: `user_id=eq.${user.id}`
          },
          (payload) => {
            console.log('Wallet updated:', payload);
            setWallet(payload.new as Wallet);
            toast.success("Balance updated!");
          }
        )
        .on(
          'postgres_changes',
          {
            event: '*',
            schema: 'public',
            table: 'payment_requests',
            filter: `requester_id=eq.${user.id}`
          },
          () => {
            fetchPaymentRequests();
          }
        )
        .on(
          'postgres_changes',
          {
            event: '*',
            schema: 'public',
            table: 'payment_requests',
            filter: `recipient_id=eq.${user.id}`
          },
          () => {
            fetchPaymentRequests();
            toast.info("New payment request received!");
          }
        )
        .subscribe();

      return () => {
        supabase.removeChannel(channel);
      };
    };

    setupRealtimeSubscription();
  }, []);

  const fetchWallet = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data, error } = await supabase
        .from("wallets")
        .select("*")
        .eq("user_id", user.id)
        .maybeSingle();

      if (error) throw error;

      if (!data) {
        // Create wallet if doesn't exist
        const { data: newWallet, error: createError } = await supabase
          .from("wallets")
          .insert({ user_id: user.id, balance: 0, currency: "BDT" })
          .select()
          .single();

        if (createError) throw createError;
        setWallet(newWallet);
      } else {
        setWallet(data);
      }
    } catch (error: any) {
      console.error("Error fetching wallet:", error);
      toast.error("Failed to load wallet");
    } finally {
      setIsLoading(false);
    }
  };

  const fetchTransactions = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from("transactions")
        .select(`
          *,
          sender_profile:profiles!transactions_sender_id_fkey(full_name, phone),
          recipient_profile:profiles!transactions_recipient_id_fkey(full_name, phone)
        `)
        .or(`sender_id.eq.${user.id},recipient_id.eq.${user.id}`)
        .order("created_at", { ascending: false })
        .limit(50);

      if (error) throw error;
      setTransactions(data || []);
    } catch (error: any) {
      console.error("Error fetching transactions:", error);
    }
  };

  const fetchPaymentRequests = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from("payment_requests")
        .select(`
          *,
          requester_profile:profiles!payment_requests_requester_id_fkey(full_name, phone),
          recipient_profile:profiles!payment_requests_recipient_id_fkey(full_name, phone)
        `)
        .or(`requester_id.eq.${user.id},recipient_id.eq.${user.id}`)
        .order("created_at", { ascending: false });

      if (error) throw error;
      setPaymentRequests(data || []);
    } catch (error: any) {
      console.error("Error fetching payment requests:", error);
    }
  };

  const handleSendMoney = async () => {
    try {
      // Validate inputs
      const validation = sendMoneySchema.safeParse({
        recipientPhone,
        amount: parseFloat(amount),
        note
      });

      if (!validation.success) {
        toast.error(validation.error.errors[0].message);
        return;
      }

      if (!wallet || wallet.balance < validation.data.amount) {
        toast.error("Insufficient balance");
        return;
      }

      setIsSending(true);

      const { data, error } = await supabase.functions.invoke('smart-pay-transfer', {
        body: {
          recipientPhone: validation.data.recipientPhone,
          amount: validation.data.amount,
          note: validation.data.note || ""
        }
      });

      if (error) throw error;

      if (data.success) {
        toast.success(`৳${validation.data.amount} sent successfully`);
        setRecipientPhone("");
        setAmount("");
        setNote("");
        fetchWallet();
        fetchTransactions();
      } else {
        toast.error(data.error || "Failed to send money");
      }
    } catch (error: any) {
      console.error("Error sending money:", error);
      toast.error(error.message || "Failed to send money");
    } finally {
      setIsSending(false);
    }
  };

  const handleRequestMoney = async () => {
    try {
      const validation = requestMoneySchema.safeParse({
        recipientPhone: requestRecipientPhone,
        amount: parseFloat(requestAmount),
        note: requestNote
      });

      if (!validation.success) {
        toast.error(validation.error.errors[0].message);
        return;
      }

      if (!wallet) {
        toast.error("Wallet not loaded");
        return;
      }

      setIsRequesting(true);

      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      // Find recipient by phone
      const { data: recipientProfile, error: profileError } = await supabase
        .from("profiles")
        .select("id")
        .eq("phone", validation.data.recipientPhone)
        .maybeSingle();

      if (profileError) throw profileError;
      if (!recipientProfile) {
        toast.error("Recipient not found");
        return;
      }

      if (recipientProfile.id === user.id) {
        toast.error("Cannot request money from yourself");
        return;
      }

      // Create payment request
      const { error: insertError } = await supabase
        .from("payment_requests")
        .insert({
          requester_id: user.id,
          recipient_id: recipientProfile.id,
          amount: validation.data.amount,
          note: validation.data.note || null,
          status: 'pending'
        });

      if (insertError) throw insertError;

      toast.success(`Payment request sent for ৳${validation.data.amount}`);
      setRequestRecipientPhone("");
      setRequestAmount("");
      setRequestNote("");
      fetchPaymentRequests();
    } catch (error: any) {
      console.error("Error requesting money:", error);
      toast.error(error.message || "Failed to send payment request");
    } finally {
      setIsRequesting(false);
    }
  };

  const handlePaymentRequestAction = async (requestId: string, action: 'accept' | 'reject' | 'cancel') => {
    try {
      const { data, error } = await supabase.functions.invoke('handle-payment-request', {
        body: { requestId, action }
      });

      if (error) throw error;

      if (data.success) {
        const actionText = action === 'accept' ? 'accepted' : action === 'reject' ? 'rejected' : 'cancelled';
        toast.success(`Payment request ${actionText}`);
        fetchPaymentRequests();
        if (action === 'accept') {
          fetchWallet();
          fetchTransactions();
        }
      } else {
        toast.error(data.error || `Failed to ${action} payment request`);
      }
    } catch (error: any) {
      console.error(`Error ${action}ing payment request:`, error);
      toast.error(error.message || `Failed to ${action} payment request`);
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-BD', {
      style: 'currency',
      currency: 'BDT',
      minimumFractionDigits: 0
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString('en-US', {
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const getTransactionType = (transaction: Transaction, userId: string) => {
    return transaction.sender_id === userId ? 'sent' : 'received';
  };

  // Filter and search transactions
  const filteredTransactions = useMemo(() => {
    if (!wallet) return [];
    
    return transactions.filter((tx) => {
      const isSent = tx.sender_id === wallet.user_id;
      const transactionType = isSent ? 'sent' : 'received';
      
      // Search filter
      if (searchQuery) {
        const query = searchQuery.toLowerCase();
        const senderName = tx.sender_profile?.full_name?.toLowerCase() || '';
        const recipientName = tx.recipient_profile?.full_name?.toLowerCase() || '';
        const note = tx.note?.toLowerCase() || '';
        const amount = tx.amount.toString();
        
        const matchesSearch = 
          senderName.includes(query) ||
          recipientName.includes(query) ||
          note.includes(query) ||
          amount.includes(query);
        
        if (!matchesSearch) return false;
      }
      
      // Transaction type filter
      if (filterType !== 'all') {
        if (filterType !== transactionType) return false;
      }
      
      // Status filter
      if (filterStatus !== 'all') {
        if (tx.status !== filterStatus) return false;
      }
      
      // Amount range filter
      if (minAmount && tx.amount < parseFloat(minAmount)) return false;
      if (maxAmount && tx.amount > parseFloat(maxAmount)) return false;
      
      // Date range filter
      if (startDate) {
        const txDate = new Date(tx.created_at);
        const filterStartDate = new Date(startDate);
        filterStartDate.setHours(0, 0, 0, 0);
        if (txDate < filterStartDate) return false;
      }
      
      if (endDate) {
        const txDate = new Date(tx.created_at);
        const filterEndDate = new Date(endDate);
        filterEndDate.setHours(23, 59, 59, 999);
        if (txDate > filterEndDate) return false;
      }
      
      return true;
    });
  }, [transactions, wallet, searchQuery, filterType, filterStatus, minAmount, maxAmount, startDate, endDate]);

  const clearFilters = () => {
    setSearchQuery("");
    setFilterType("all");
    setFilterStatus("all");
    setMinAmount("");
    setMaxAmount("");
    setStartDate("");
    setEndDate("");
  };

  const hasActiveFilters = searchQuery || filterType !== 'all' || filterStatus !== 'all' || 
    minAmount || maxAmount || startDate || endDate;

  const openTransactionDetails = (transaction: Transaction) => {
    setSelectedTransaction(transaction);
    setIsDetailsModalOpen(true);
  };

  const closeTransactionDetails = () => {
    setIsDetailsModalOpen(false);
    setSelectedTransaction(null);
    setNewTag("");
    setIsAddingTag(false);
  };

  const handleAddTag = async () => {
    if (!selectedTransaction || !newTag.trim()) return;

    try {
      setIsAddingTag(true);
      const currentTags = selectedTransaction.tags || [];
      const updatedTags = [...currentTags, newTag.trim()];

      const { error } = await supabase
        .from("transactions")
        .update({ tags: updatedTags })
        .eq("id", selectedTransaction.id);

      if (error) throw error;

      // Update local state
      setSelectedTransaction({ ...selectedTransaction, tags: updatedTags });
      setTransactions(transactions.map(tx => 
        tx.id === selectedTransaction.id ? { ...tx, tags: updatedTags } : tx
      ));
      setNewTag("");
      toast.success("Tag added successfully");
    } catch (error: any) {
      console.error("Error adding tag:", error);
      toast.error("Failed to add tag");
    } finally {
      setIsAddingTag(false);
    }
  };

  const handleRemoveTag = async (tagToRemove: string) => {
    if (!selectedTransaction) return;

    try {
      const currentTags = selectedTransaction.tags || [];
      const updatedTags = currentTags.filter(tag => tag !== tagToRemove);

      const { error } = await supabase
        .from("transactions")
        .update({ tags: updatedTags })
        .eq("id", selectedTransaction.id);

      if (error) throw error;

      // Update local state
      setSelectedTransaction({ ...selectedTransaction, tags: updatedTags });
      setTransactions(transactions.map(tx => 
        tx.id === selectedTransaction.id ? { ...tx, tags: updatedTags } : tx
      ));
      toast.success("Tag removed successfully");
    } catch (error: any) {
      console.error("Error removing tag:", error);
      toast.error("Failed to remove tag");
    }
  };

  return (
    <div className="h-full overflow-auto pb-20">
      <div className="p-4 sm:p-6 max-w-4xl mx-auto space-y-6">
        {/* Header */}
        <div className="flex items-center gap-3">
          <div className="relative">
            <div className="absolute inset-0 bg-gradient-to-br from-green-500 to-blue-500 rounded-lg blur-lg opacity-40" />
            <div className="relative bg-gradient-to-br from-green-500 to-blue-500 p-2 rounded-lg">
              <Wallet className="w-6 h-6 text-white" />
            </div>
          </div>
          <div>
            <h1 className="text-2xl sm:text-3xl font-bold">Smart Pay</h1>
            <p className="text-sm text-muted-foreground">Send money and manage payments</p>
          </div>
        </div>

        {/* Wallet Balance Card */}
        <Card>
          <CardHeader className="pb-3">
            <div className="flex items-center justify-between">
              <CardTitle className="text-lg">Wallet Balance</CardTitle>
              <Button
                size="sm"
                variant="ghost"
                onClick={fetchWallet}
              >
                <RefreshCw className="w-4 h-4" />
              </Button>
            </div>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="text-center py-4 text-muted-foreground">Loading...</div>
            ) : (
              <div>
                <p className="text-4xl font-bold">
                  {wallet ? formatCurrency(wallet.balance) : '৳0'}
                </p>
                <p className="text-sm text-muted-foreground mt-1">
                  Available Balance
                </p>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Main Tabs */}
        <Tabs defaultValue="send" className="w-full">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="send">Send</TabsTrigger>
            <TabsTrigger value="request">Request</TabsTrigger>
            <TabsTrigger value="history">History</TabsTrigger>
          </TabsList>

          {/* Send Money Tab */}
          <TabsContent value="send" className="space-y-4 mt-4">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Send className="w-5 h-5" />
                  Send Money
                </CardTitle>
                <CardDescription>Transfer money to another Smart Friend user</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <label className="text-sm font-medium">Recipient Phone Number</label>
                  <Input
                    type="tel"
                    placeholder="01XXXXXXXXX"
                    value={recipientPhone}
                    onChange={(e) => setRecipientPhone(e.target.value)}
                    maxLength={15}
                  />
                </div>

                <div className="space-y-2">
                  <label className="text-sm font-medium">Amount (BDT)</label>
                  <Input
                    type="number"
                    placeholder="0"
                    value={amount}
                    onChange={(e) => setAmount(e.target.value)}
                    min="1"
                    max="1000000"
                  />
                </div>

                <div className="space-y-2">
                  <label className="text-sm font-medium">Note (Optional)</label>
                  <Input
                    type="text"
                    placeholder="Payment for..."
                    value={note}
                    onChange={(e) => setNote(e.target.value)}
                    maxLength={200}
                  />
                </div>

                <Button
                  onClick={handleSendMoney}
                  disabled={isSending || !recipientPhone || !amount}
                  className="w-full"
                >
                  <DollarSign className="w-4 h-4 mr-2" />
                  {isSending ? "Sending..." : "Send Money"}
                </Button>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Request Money Tab */}
          <TabsContent value="request" className="space-y-4 mt-4">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <DollarSign className="w-5 h-5" />
                  Request Money
                </CardTitle>
                <CardDescription>Request payment from another Smart Friend user</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <label className="text-sm font-medium">From Phone Number</label>
                  <Input
                    type="tel"
                    placeholder="01XXXXXXXXX"
                    value={requestRecipientPhone}
                    onChange={(e) => setRequestRecipientPhone(e.target.value)}
                    maxLength={15}
                  />
                </div>

                <div className="space-y-2">
                  <label className="text-sm font-medium">Amount (BDT)</label>
                  <Input
                    type="number"
                    placeholder="0"
                    value={requestAmount}
                    onChange={(e) => setRequestAmount(e.target.value)}
                    min="1"
                    max="1000000"
                  />
                </div>

                <div className="space-y-2">
                  <label className="text-sm font-medium">Note (Optional)</label>
                  <Input
                    type="text"
                    placeholder="Payment request for..."
                    value={requestNote}
                    onChange={(e) => setRequestNote(e.target.value)}
                    maxLength={200}
                  />
                </div>

                <Button
                  onClick={handleRequestMoney}
                  disabled={isRequesting || !requestRecipientPhone || !requestAmount}
                  className="w-full"
                >
                  <Send className="w-4 h-4 mr-2" />
                  {isRequesting ? "Sending Request..." : "Send Request"}
                </Button>
              </CardContent>
            </Card>

            {/* Payment Requests List */}
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">Payment Requests</CardTitle>
                <CardDescription>
                  {paymentRequests.length} requests found
                </CardDescription>
              </CardHeader>
              <CardContent>
                {paymentRequests.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    No payment requests
                  </div>
                ) : (
                  <div className="space-y-3">
                    {paymentRequests.map((request) => {
                      const isRequester = wallet && request.requester_id === wallet.user_id;
                      const isPending = request.status === 'pending';
                      
                      return (
                        <div
                          key={request.id}
                          className="flex items-start justify-between p-3 rounded-lg bg-muted/50"
                        >
                          <div className="flex-1">
                            <div className="flex items-center gap-2 mb-1">
                              <p className="font-medium text-sm">
                                {isRequester ? 'To' : 'From'}{' '}
                                {isRequester
                                  ? request.recipient_profile?.full_name
                                  : request.requester_profile?.full_name}
                              </p>
                              <Badge 
                                variant={
                                  request.status === 'accepted' ? 'secondary' : 
                                  request.status === 'rejected' ? 'destructive' : 
                                  request.status === 'cancelled' ? 'outline' : 
                                  'default'
                                }
                                className="text-xs"
                              >
                                {request.status}
                              </Badge>
                            </div>
                            <p className="text-lg font-semibold text-primary">
                              ৳{request.amount.toLocaleString()}
                            </p>
                            {request.note && (
                              <p className="text-xs text-muted-foreground mt-1">{request.note}</p>
                            )}
                            <p className="text-xs text-muted-foreground mt-1">
                              {formatDate(request.created_at)}
                            </p>
                            
                            {/* Action Buttons */}
                            {isPending && (
                              <div className="flex gap-2 mt-2">
                                {isRequester ? (
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    onClick={() => handlePaymentRequestAction(request.id, 'cancel')}
                                  >
                                    Cancel
                                  </Button>
                                ) : (
                                  <>
                                    <Button
                                      size="sm"
                                      variant="default"
                                      onClick={() => handlePaymentRequestAction(request.id, 'accept')}
                                    >
                                      Accept & Pay
                                    </Button>
                                    <Button
                                      size="sm"
                                      variant="outline"
                                      onClick={() => handlePaymentRequestAction(request.id, 'reject')}
                                    >
                                      Reject
                                    </Button>
                                  </>
                                )}
                              </div>
                            )}
                          </div>
                        </div>
                      );
                    })}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Transaction History Tab */}
          <TabsContent value="history" className="space-y-4 mt-4">
            {/* Search and Filters */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle className="flex items-center gap-2">
                    <Search className="w-5 h-5" />
                    Search & Filter
                  </CardTitle>
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => setShowFilters(!showFilters)}
                  >
                    <Filter className="w-4 h-4 mr-2" />
                    {showFilters ? 'Hide' : 'Show'} Filters
                  </Button>
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                {/* Search Bar */}
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                  <Input
                    type="text"
                    placeholder="Search by name, note, or amount..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    className="pl-10"
                  />
                  {searchQuery && (
                    <Button
                      size="sm"
                      variant="ghost"
                      className="absolute right-1 top-1/2 -translate-y-1/2 h-7 w-7 p-0"
                      onClick={() => setSearchQuery("")}
                    >
                      <X className="w-4 h-4" />
                    </Button>
                  )}
                </div>

                {/* Advanced Filters */}
                {showFilters && (
                  <div className="space-y-4 pt-4 border-t">
                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      {/* Transaction Type */}
                      <div className="space-y-2">
                        <label className="text-sm font-medium">Transaction Type</label>
                        <Select value={filterType} onValueChange={setFilterType}>
                          <SelectTrigger>
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="all">All Types</SelectItem>
                            <SelectItem value="sent">Sent</SelectItem>
                            <SelectItem value="received">Received</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>

                      {/* Status */}
                      <div className="space-y-2">
                        <label className="text-sm font-medium">Status</label>
                        <Select value={filterStatus} onValueChange={setFilterStatus}>
                          <SelectTrigger>
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="all">All Status</SelectItem>
                            <SelectItem value="completed">Completed</SelectItem>
                            <SelectItem value="pending">Pending</SelectItem>
                            <SelectItem value="failed">Failed</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                    </div>

                    {/* Amount Range */}
                    <div className="space-y-2">
                      <label className="text-sm font-medium">Amount Range (BDT)</label>
                      <div className="grid grid-cols-2 gap-2">
                        <Input
                          type="number"
                          placeholder="Min"
                          value={minAmount}
                          onChange={(e) => setMinAmount(e.target.value)}
                          min="0"
                        />
                        <Input
                          type="number"
                          placeholder="Max"
                          value={maxAmount}
                          onChange={(e) => setMaxAmount(e.target.value)}
                          min="0"
                        />
                      </div>
                    </div>

                    {/* Date Range */}
                    <div className="space-y-2">
                      <label className="text-sm font-medium">Date Range</label>
                      <div className="grid grid-cols-2 gap-2">
                        <Input
                          type="date"
                          value={startDate}
                          onChange={(e) => setStartDate(e.target.value)}
                        />
                        <Input
                          type="date"
                          value={endDate}
                          onChange={(e) => setEndDate(e.target.value)}
                        />
                      </div>
                    </div>

                    {/* Clear Filters */}
                    {hasActiveFilters && (
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={clearFilters}
                        className="w-full"
                      >
                        <X className="w-4 h-4 mr-2" />
                        Clear All Filters
                      </Button>
                    )}
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Transaction List */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <History className="w-5 h-5" />
                  Transaction History
                </CardTitle>
                <CardDescription>
                  {filteredTransactions.length} of {transactions.length} transactions
                  {hasActiveFilters && ' (filtered)'}
                </CardDescription>
              </CardHeader>
              <CardContent>
                {filteredTransactions.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    {transactions.length === 0 ? 'No transactions yet' : 'No transactions match your filters'}
                  </div>
                ) : (
                  <div className="space-y-3">
                    {filteredTransactions.map((tx) => {
                      const isSent = wallet && tx.sender_id === wallet.user_id;
                      
                      return (
                        <div
                          key={tx.id}
                          className="flex items-center justify-between p-3 rounded-lg bg-muted/50 hover:bg-muted transition-colors cursor-pointer group"
                          onClick={() => openTransactionDetails(tx)}
                        >
                          <div className="flex items-center gap-3 flex-1">
                            <div className={`p-2 rounded-full ${isSent ? 'bg-red-500/10' : 'bg-green-500/10'}`}>
                              {isSent ? (
                                <ArrowUpRight className="w-4 h-4 text-red-500" />
                              ) : (
                                <ArrowDownRight className="w-4 h-4 text-green-500" />
                              )}
                            </div>
                            <div className="flex-1">
                              <p className="font-medium text-sm">
                                {isSent ? 'Sent to' : 'Received from'}{' '}
                                {isSent
                                  ? tx.recipient_profile?.full_name
                                  : tx.sender_profile?.full_name}
                              </p>
                              {tx.note && (
                                <p className="text-xs text-muted-foreground">{tx.note}</p>
                              )}
                              {tx.tags && tx.tags.length > 0 && (
                                <div className="flex gap-1 mt-1 flex-wrap">
                                  {tx.tags.slice(0, 2).map((tag, idx) => (
                                    <Badge key={idx} variant="outline" className="text-xs">
                                      {tag}
                                    </Badge>
                                  ))}
                                  {tx.tags.length > 2 && (
                                    <Badge variant="outline" className="text-xs">
                                      +{tx.tags.length - 2}
                                    </Badge>
                                  )}
                                </div>
                              )}
                              <p className="text-xs text-muted-foreground mt-1">
                                {formatDate(tx.created_at)}
                              </p>
                            </div>
                          </div>
                          <div className="text-right flex items-center gap-2">
                            <div>
                              <p className={`font-semibold ${isSent ? 'text-red-500' : 'text-green-500'}`}>
                                {isSent ? '-' : '+'}{formatCurrency(tx.amount)}
                              </p>
                              <Badge variant={tx.status === 'completed' ? 'secondary' : 'outline'} className="text-xs">
                                {tx.status}
                              </Badge>
                            </div>
                            <Info className="w-4 h-4 text-muted-foreground opacity-0 group-hover:opacity-100 transition-opacity" />
                          </div>
                        </div>
                      );
                    })}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        {/* Transaction Details Modal */}
        <Dialog open={isDetailsModalOpen} onOpenChange={setIsDetailsModalOpen}>
          <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2">
                <Info className="w-5 h-5" />
                Transaction Details
              </DialogTitle>
              <DialogDescription>
                Complete information about this transaction
              </DialogDescription>
            </DialogHeader>

            {selectedTransaction && wallet && (
              <div className="space-y-6 mt-4">
                {/* Transaction Type Header */}
                <div className="flex items-center gap-3 p-4 rounded-lg bg-muted/50">
                  <div className={`p-3 rounded-full ${
                    selectedTransaction.sender_id === wallet.user_id 
                      ? 'bg-red-500/10' 
                      : 'bg-green-500/10'
                  }`}>
                    {selectedTransaction.sender_id === wallet.user_id ? (
                      <ArrowUpRight className="w-6 h-6 text-red-500" />
                    ) : (
                      <ArrowDownRight className="w-6 h-6 text-green-500" />
                    )}
                  </div>
                  <div className="flex-1">
                    <h3 className="font-semibold text-lg">
                      {selectedTransaction.sender_id === wallet.user_id ? 'Sent' : 'Received'}
                    </h3>
                    <p className={`text-2xl font-bold ${
                      selectedTransaction.sender_id === wallet.user_id 
                        ? 'text-red-500' 
                        : 'text-green-500'
                    }`}>
                      {selectedTransaction.sender_id === wallet.user_id ? '-' : '+'}
                      {formatCurrency(selectedTransaction.amount)}
                    </p>
                  </div>
                  <Badge 
                    variant={selectedTransaction.status === 'completed' ? 'secondary' : 'outline'}
                  >
                    {selectedTransaction.status}
                  </Badge>
                </div>

                {/* Transaction Information */}
                <div className="space-y-3">
                  <h4 className="font-semibold text-sm text-muted-foreground uppercase">
                    Transaction Information
                  </h4>
                  
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-xs text-muted-foreground">Transaction ID</p>
                      <p className="text-sm font-mono break-all">{selectedTransaction.id}</p>
                    </div>
                    
                    <div>
                      <p className="text-xs text-muted-foreground">Type</p>
                      <p className="text-sm capitalize">{selectedTransaction.type}</p>
                    </div>
                    
                    <div>
                      <p className="text-xs text-muted-foreground">
                        {selectedTransaction.sender_id === wallet.user_id ? 'Recipient' : 'Sender'}
                      </p>
                      <p className="text-sm">
                        {selectedTransaction.sender_id === wallet.user_id
                          ? selectedTransaction.recipient_profile?.full_name
                          : selectedTransaction.sender_profile?.full_name}
                      </p>
                      <p className="text-xs text-muted-foreground">
                        {selectedTransaction.sender_id === wallet.user_id
                          ? selectedTransaction.recipient_profile?.phone
                          : selectedTransaction.sender_profile?.phone}
                      </p>
                    </div>
                    
                    <div>
                      <p className="text-xs text-muted-foreground">Amount</p>
                      <p className="text-sm font-semibold">
                        ৳{selectedTransaction.amount.toLocaleString()}
                      </p>
                    </div>
                  </div>
                </div>

                {/* Note */}
                {selectedTransaction.note && (
                  <div className="space-y-2">
                    <h4 className="font-semibold text-sm text-muted-foreground uppercase">
                      Note
                    </h4>
                    <p className="text-sm bg-muted/50 p-3 rounded-lg">
                      {selectedTransaction.note}
                    </p>
                  </div>
                )}

                {/* Timestamps */}
                <div className="space-y-3">
                  <h4 className="font-semibold text-sm text-muted-foreground uppercase">
                    Timestamps
                  </h4>
                  
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-xs text-muted-foreground">Created At</p>
                      <p className="text-sm">
                        {new Date(selectedTransaction.created_at).toLocaleString('en-US', {
                          dateStyle: 'medium',
                          timeStyle: 'short'
                        })}
                      </p>
                    </div>
                    
                    <div>
                      <p className="text-xs text-muted-foreground">Updated At</p>
                      <p className="text-sm">
                        {new Date(selectedTransaction.updated_at).toLocaleString('en-US', {
                          dateStyle: 'medium',
                          timeStyle: 'short'
                        })}
                      </p>
                    </div>
                  </div>
                </div>

                {/* Tags / Categories */}
                <div className="space-y-3">
                  <h4 className="font-semibold text-sm text-muted-foreground uppercase flex items-center gap-2">
                    <Tag className="w-4 h-4" />
                    Tags / Categories
                  </h4>
                  
                  {/* Existing Tags */}
                  {selectedTransaction.tags && selectedTransaction.tags.length > 0 && (
                    <div className="flex flex-wrap gap-2">
                      {selectedTransaction.tags.map((tag, idx) => (
                        <Badge 
                          key={idx} 
                          variant="secondary"
                          className="gap-1 pr-1"
                        >
                          {tag}
                          <Button
                            size="sm"
                            variant="ghost"
                            className="h-4 w-4 p-0 hover:bg-destructive hover:text-destructive-foreground"
                            onClick={() => handleRemoveTag(tag)}
                          >
                            <X className="w-3 h-3" />
                          </Button>
                        </Badge>
                      ))}
                    </div>
                  )}
                  
                  {/* Add New Tag */}
                  <div className="flex gap-2">
                    <Input
                      type="text"
                      placeholder="Add a tag..."
                      value={newTag}
                      onChange={(e) => setNewTag(e.target.value)}
                      onKeyPress={(e) => e.key === 'Enter' && handleAddTag()}
                      maxLength={30}
                    />
                    <Button
                      size="sm"
                      onClick={handleAddTag}
                      disabled={isAddingTag || !newTag.trim()}
                    >
                      <Plus className="w-4 h-4 mr-1" />
                      Add
                    </Button>
                  </div>
                </div>

                {/* Close Button */}
                <div className="flex justify-end pt-4 border-t">
                  <Button onClick={closeTransactionDetails} variant="outline">
                    Close
                  </Button>
                </div>
              </div>
            )}
          </DialogContent>
        </Dialog>
      </div>
    </div>
  );
};

export default SmartPay;
