import { useState } from "react";
import { 
  Rocket, Code, Database, Cloud, Terminal, Wallet, BarChart3, Users, Search, Settings, 
  Sparkles, Megaphone, Palette, BrainCircuit, Phone, Globe, ListTodo, Bug, FileText,
  Home, Wrench, Copy, Cpu, Package, CreditCard, Upload, Share2, User, FileCode
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { cn } from "@/lib/utils";
import { useWallet } from "@/hooks/useWallet";
import { useNavigate } from "react-router-dom";

type ViewType = "home" | "ide" | "devops" | "call" | "clone" | "ai-engine" | "search" | "modules" | "designer" | "memory" | "wallet" | "subscription" | "deploy" | "affiliate" | "profile";

const Studio = () => {
  const [activeView, setActiveView] = useState<ViewType>("home");
  const { wallet } = useWallet();
  const navigate = useNavigate();

  const menuItems = [
    { id: "home" as ViewType, label: "Home", icon: Home, route: "/studio" },
    { id: "ide" as ViewType, label: "IDE", icon: Code, route: "/studio/ide" },
    { id: "devops" as ViewType, label: "DevOps", icon: Wrench, route: "/studio/devops" },
    { id: "call" as ViewType, label: "AI Call Center", icon: Phone, route: "/studio/call" },
    { id: "clone" as ViewType, label: "Website Cloner", icon: Copy, route: "/studio/clone" },
    { id: "ai-engine" as ViewType, label: "AI Engine", icon: Cpu, route: "/studio/ai-engine" },
    { id: "search" as ViewType, label: "Search Engine", icon: Sparkles, route: "/studio/search" },
    { id: "modules" as ViewType, label: "Modules", icon: Package, route: "/studio/modules" },
    { id: "designer" as ViewType, label: "Website Designer", icon: Palette, route: "/studio/designer" },
    { id: "memory" as ViewType, label: "Memory", icon: Database, route: "/studio/memory" },
    { id: "wallet" as ViewType, label: "Wallet", icon: Wallet, route: "/studio/wallet" },
    { id: "subscription" as ViewType, label: "Subscription", icon: CreditCard, route: "/studio/subscription" },
    { id: "deploy" as ViewType, label: "Deploy", icon: Upload, route: "/studio/deploy" },
    { id: "affiliate" as ViewType, label: "Affiliate", icon: Share2, route: "/studio/affiliate" },
    { id: "profile" as ViewType, label: "Profile", icon: User, route: "/profile" },
  ];

  const newFeatures = [
    {
      title: "AI Task Engine",
      description: "Autonomous task management and execution",
      icon: ListTodo,
      color: "from-indigo-500 to-purple-600",
      route: "/studio/tasks"
    },
    {
      title: "Error Dashboard",
      description: "Auto bug finder and fixer",
      icon: Bug,
      color: "from-red-500 to-pink-600",
      route: "/studio/errors"
    },
    {
      title: "Auto Documentation",
      description: "Generate comprehensive project docs",
      icon: FileText,
      color: "from-slate-500 to-gray-600",
      route: "/studio/docs"
    },
    {
      title: "Command Center",
      description: "Smart Friend-1 Boss Assistant",
      icon: Terminal,
      color: "from-blue-500 to-cyan-600",
      route: "/studio/command"
    },
  ];

  const handleNavigation = (item: typeof menuItems[0]) => {
    setActiveView(item.id);
    navigate(item.route);
  };

  return (
    <div className="flex h-screen bg-gradient-to-br from-primary/5 via-background to-secondary/5">
      <aside className="w-64 bg-card/95 backdrop-blur-sm border-r border-border flex flex-col shadow-xl">
        <div className="p-6 border-b border-border/50 bg-gradient-to-r from-primary/10 to-secondary/10">
          <div className="flex items-center gap-3">
            <div className="p-2 bg-primary/20 rounded-xl">
              <FileCode className="h-6 w-6 text-primary" />
            </div>
            <div>
              <h2 className="text-lg font-bold bg-gradient-to-r from-primary to-secondary bg-clip-text text-transparent">
                Smart Friend
              </h2>
              <p className="text-xs text-muted-foreground">ULTRA v2.0</p>
            </div>
          </div>
        </div>

        <div className="flex items-center justify-between px-4 py-3 bg-gradient-to-r from-green-500/10 to-emerald-500/10 border-b border-border/50">
          <span className="text-xs font-medium text-muted-foreground">Balance</span>
          <div className="flex items-center gap-2">
            <Sparkles className="h-4 w-4 text-yellow-500" />
            <span className="text-sm font-bold text-foreground">{wallet?.balance || 0}</span>
          </div>
        </div>

        <nav className="flex-1 p-4 space-y-2 overflow-y-auto">
          {menuItems.map((item) => {
            const Icon = item.icon;
            return (
              <Button
                key={item.id}
                onClick={() => handleNavigation(item)}
                variant={activeView === item.id ? "default" : "ghost"}
                className={cn(
                  "w-full justify-start gap-3 transition-all duration-200",
                  activeView === item.id 
                    ? "bg-gradient-to-r from-primary to-primary/80 text-primary-foreground shadow-lg scale-105" 
                    : "hover:bg-primary/10 hover:scale-105"
                )}
              >
                <Icon className="h-5 w-5" />
                <span>{item.label}</span>
              </Button>
            );
          })}
        </nav>

        <div className="p-4 border-t border-border/50 bg-gradient-to-r from-primary/5 to-secondary/5">
          <div className="text-xs text-muted-foreground text-center space-y-1">
            <p className="font-semibold">Smart Friend ULTRA v2.0</p>
            <p>© 2025 Smart Seba Ecosystem</p>
          </div>
        </div>
      </aside>

      <main className="flex-1 overflow-hidden">
        <div className="h-full flex items-center justify-center p-8">
          <div className="text-center space-y-6 max-w-2xl">
            <div className="inline-block p-4 bg-gradient-to-br from-primary/20 to-secondary/20 rounded-2xl">
              <FileCode className="h-16 w-16 text-primary" />
            </div>
            <h1 className="text-4xl font-bold bg-gradient-to-r from-primary via-purple-500 to-secondary bg-clip-text text-transparent">
              Welcome to Smart Friend ULTRA
            </h1>
            <p className="text-lg text-muted-foreground">
              Your all-in-one AI-powered development studio
            </p>
            <div className="flex flex-wrap gap-3 justify-center pt-4">
              <Button onClick={() => handleNavigation(menuItems[1])} className="gap-2">
                <Code className="h-4 w-4" />
                Open IDE
              </Button>
              <Button onClick={() => handleNavigation(menuItems[2])} variant="outline" className="gap-2">
                <Cpu className="h-4 w-4" />
                AI Engine
              </Button>
              <Button onClick={() => handleNavigation(menuItems[3])} variant="outline" className="gap-2">
                <Sparkles className="h-4 w-4" />
                Search Engine
              </Button>
              <Button onClick={() => handleNavigation(menuItems[4])} variant="outline" className="gap-2">
                <Package className="h-4 w-4" />
                Create Module
              </Button>
            </div>
          </div>
        </div>
      </main>
    </div>
  );
};

export default Studio;
