import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { useAIBuilder } from "@/hooks/useAIBuilder";
import { WalletGuard } from "@/components/ide/WalletGuard";
import { Loader2, Sparkles, Code, FileCode } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { cn } from "@/lib/utils";

const StudioAIEngine = () => {
  const [projectId] = useState("default-project");
  const [prompt, setPrompt] = useState("");
  const [filePath, setFilePath] = useState("");
  const [isGenerating, setIsGenerating] = useState(false);
  const [useWebSearch, setUseWebSearch] = useState(false);
  const { generateCode, changes } = useAIBuilder(projectId);
  const { toast } = useToast();

  const handleGenerate = async () => {
    if (!prompt.trim()) {
      toast({ title: "Error", description: "Please enter a prompt", variant: "destructive" });
      return;
    }
    if (!filePath.trim()) {
      toast({ title: "Error", description: "Please enter a file path", variant: "destructive" });
      return;
    }

    setIsGenerating(true);
    try {
      await generateCode.mutateAsync({ prompt, filePath, originalCode: "" });
      setPrompt("");
      setFilePath("");
    } catch (error) {
      console.error("Generation error:", error);
    } finally {
      setIsGenerating(false);
    }
  };

  return (
    <WalletGuard>
      <div className="h-screen bg-gradient-to-br from-primary/5 via-background to-secondary/5 p-6 overflow-y-auto">
        <div className="max-w-6xl mx-auto space-y-6">
          <div className="flex items-center gap-3">
            <div className="p-3 bg-gradient-to-br from-primary/20 to-secondary/20 rounded-xl">
              <Sparkles className="h-8 w-8 text-primary" />
            </div>
            <div>
              <h1 className="text-3xl font-bold bg-gradient-to-r from-primary to-secondary bg-clip-text text-transparent">
                AI Code Engine
              </h1>
              <p className="text-muted-foreground">Generate code with natural language</p>
            </div>
          </div>

          <Card className="shadow-xl">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Code className="h-5 w-5" />
                Code Generator
              </CardTitle>
              <CardDescription>
                Describe what you want to build and AI will generate the code
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label>File Path</Label>
                <Input
                  placeholder="e.g., src/components/LoginForm.tsx"
                  value={filePath}
                  onChange={(e) => setFilePath(e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label>What do you want to build?</Label>
                <Textarea
                  placeholder="e.g., Create a login form with email and password fields, validation, and a submit button"
                  value={prompt}
                  onChange={(e) => setPrompt(e.target.value)}
                  rows={6}
                  className="resize-none"
                />
              </div>
              <div className="flex items-center gap-4">
                <div className="flex items-center gap-2">
                  <Switch
                    id="web-search"
                    checked={useWebSearch}
                    onCheckedChange={setUseWebSearch}
                  />
                  <Label htmlFor="web-search" className="text-sm cursor-pointer">
                    Use Web Search
                  </Label>
                </div>
              </div>
              <Button 
                onClick={handleGenerate} 
                disabled={isGenerating || !prompt.trim() || !filePath.trim()}
                className="w-full gap-2"
              >
                {isGenerating ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin" />
                    Generating...
                  </>
                ) : (
                  <>
                    <Sparkles className="h-4 w-4" />
                    Generate Code
                  </>
                )}
              </Button>
            </CardContent>
          </Card>

          <Card className="shadow-xl">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileCode className="h-5 w-5" />
                Recent Changes
              </CardTitle>
              <CardDescription>
                View and manage your AI-generated code changes
              </CardDescription>
            </CardHeader>
            <CardContent>
              {changes.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  No code changes yet. Start generating!
                </div>
              ) : (
                <div className="space-y-4">
                  {changes.slice(0, 5).map((change) => (
                    <div
                      key={change.id}
                      className="p-4 border border-border rounded-lg bg-card/50 hover:bg-card transition-colors"
                    >
                      <div className="flex items-start justify-between">
                        <div className="space-y-1">
                          <p className="font-medium">{change.file_path}</p>
                          <p className="text-sm text-muted-foreground">{change.prompt}</p>
                          <p className="text-xs text-muted-foreground">
                            {new Date(change.created_at).toLocaleString()}
                          </p>
                        </div>
                        <div className={cn(
                          "px-2 py-1 rounded text-xs font-medium",
                          change.applied 
                            ? "bg-green-500/20 text-green-600" 
                            : "bg-yellow-500/20 text-yellow-600"
                        )}>
                          {change.applied ? "Applied" : "Pending"}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </WalletGuard>
  );
};

export default StudioAIEngine;
