import { useState } from "react";
import { useAffiliate } from "@/hooks/useAffiliate";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Share2, Copy, DollarSign, TrendingUp, Users, Check } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

const StudioAffiliate = () => {
  const { affiliateLink, conversions, payouts, generateLink, requestPayout } = useAffiliate();
  const { toast } = useToast();
  const [copied, setCopied] = useState(false);
  const [payoutAmount, setPayoutAmount] = useState("");
  const [paymentMethod, setPaymentMethod] = useState("bank");

  const handleGenerateLink = async () => {
    await generateLink.mutateAsync(undefined);
  };

  const handleCopyLink = () => {
    if (affiliateLink) {
      const link = `${window.location.origin}?ref=${affiliateLink.referral_code}`;
      navigator.clipboard.writeText(link);
      setCopied(true);
      toast({ title: "Link copied to clipboard" });
      setTimeout(() => setCopied(false), 2000);
    }
  };

  const handleRequestPayout = async () => {
    const amount = parseFloat(payoutAmount);
    if (isNaN(amount) || amount <= 0) {
      toast({ title: "Invalid amount", variant: "destructive" });
      return;
    }

    await requestPayout.mutateAsync({ amount, paymentMethod });
    setPayoutAmount("");
  };

  return (
    <div className="h-screen bg-gradient-to-br from-primary/5 via-background to-secondary/5 p-6 overflow-y-auto">
      <div className="max-w-6xl mx-auto space-y-6">
        <div className="flex items-center gap-3">
          <div className="p-3 bg-gradient-to-br from-primary/20 to-secondary/20 rounded-xl">
            <Share2 className="h-8 w-8 text-primary" />
          </div>
          <div>
            <h1 className="text-3xl font-bold bg-gradient-to-r from-primary to-secondary bg-clip-text text-transparent">
              Affiliate Dashboard
            </h1>
            <p className="text-muted-foreground">Earn by referring Smart Friend to others</p>
          </div>
        </div>

        <div className="grid gap-6 md:grid-cols-3">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total Earnings</CardTitle>
              <DollarSign className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">৳{affiliateLink?.total_earnings || 0}</div>
              <p className="text-xs text-muted-foreground">Available for payout</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Clicks</CardTitle>
              <TrendingUp className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{affiliateLink?.clicks || 0}</div>
              <p className="text-xs text-muted-foreground">Total referral clicks</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Conversions</CardTitle>
              <Users className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{affiliateLink?.conversions || 0}</div>
              <p className="text-xs text-muted-foreground">Successful referrals</p>
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Your Referral Link</CardTitle>
            <CardDescription>Share this link to earn commissions</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            {!affiliateLink ? (
              <Button onClick={handleGenerateLink} className="w-full">
                Generate Affiliate Link
              </Button>
            ) : (
              <div className="flex gap-2">
                <Input
                  readOnly
                  value={`${window.location.origin}?ref=${affiliateLink.referral_code}`}
                  className="font-mono text-sm"
                />
                <Button onClick={handleCopyLink} variant="outline" size="icon">
                  {copied ? <Check className="h-4 w-4" /> : <Copy className="h-4 w-4" />}
                </Button>
              </div>
            )}
          </CardContent>
        </Card>

        <div className="grid gap-6 md:grid-cols-2">
          <Card>
            <CardHeader>
              <CardTitle>Request Payout</CardTitle>
              <CardDescription>Minimum payout: ৳50</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <label className="text-sm font-medium">Amount (BDT)</label>
                <Input
                  type="number"
                  value={payoutAmount}
                  onChange={(e) => setPayoutAmount(e.target.value)}
                  placeholder="Enter amount"
                />
              </div>
              <div className="space-y-2">
                <label className="text-sm font-medium">Payment Method</label>
                <select
                  value={paymentMethod}
                  onChange={(e) => setPaymentMethod(e.target.value)}
                  className="w-full px-3 py-2 rounded-md border border-input bg-background"
                >
                  <option value="bank">Bank Transfer</option>
                  <option value="bkash">bKash</option>
                  <option value="nagad">Nagad</option>
                </select>
              </div>
              <Button onClick={handleRequestPayout} className="w-full">
                Request Payout
              </Button>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Recent Conversions</CardTitle>
              <CardDescription>Latest successful referrals</CardDescription>
            </CardHeader>
            <CardContent>
              {conversions.length === 0 ? (
                <p className="text-sm text-muted-foreground">No conversions yet</p>
              ) : (
                <div className="space-y-3">
                  {conversions.slice(0, 5).map((conversion) => (
                    <div key={conversion.id} className="flex items-center justify-between">
                      <div>
                        <p className="text-sm font-medium">৳{conversion.commission_amount}</p>
                        <p className="text-xs text-muted-foreground">
                          {new Date(conversion.created_at).toLocaleDateString()}
                        </p>
                      </div>
                      <Badge variant={conversion.status === "paid" ? "default" : "secondary"}>
                        {conversion.status}
                      </Badge>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Payout History</CardTitle>
            <CardDescription>Track your payout requests</CardDescription>
          </CardHeader>
          <CardContent>
            {payouts.length === 0 ? (
              <p className="text-sm text-muted-foreground">No payouts yet</p>
            ) : (
              <div className="space-y-3">
                {payouts.map((payout) => (
                  <div key={payout.id} className="flex items-center justify-between border-b pb-3">
                    <div>
                      <p className="text-sm font-medium">৳{payout.amount}</p>
                      <p className="text-xs text-muted-foreground">
                        {new Date(payout.created_at).toLocaleDateString()} • {payout.payment_method}
                      </p>
                    </div>
                    <Badge variant={payout.status === "processed" ? "default" : "secondary"}>
                      {payout.status}
                    </Badge>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
};

export default StudioAffiliate;