import { WalletGuard } from "@/components/ide/WalletGuard";
import { Card } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { 
  Users, TrendingUp, Wallet, Activity, 
  BarChart3, Calendar, DollarSign 
} from "lucide-react";
import { useQuery } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";

const StudioAnalytics = () => {
  const { data: stats } = useQuery({
    queryKey: ["analytics_stats"],
    queryFn: async () => {
      const [profiles, subscriptions, wallets, messages] = await Promise.all([
        supabase.from("profiles").select("id", { count: "exact", head: true }),
        supabase.from("smart_subscriptions").select("id", { count: "exact", head: true }),
        supabase.from("smart_wallets").select("balance"),
        supabase.from("messages").select("id", { count: "exact", head: true })
      ]);

      const totalBalance = wallets.data?.reduce((sum, w) => sum + w.balance, 0) || 0;

      return {
        totalUsers: profiles.count || 0,
        activeSubscriptions: subscriptions.count || 0,
        totalWalletBalance: totalBalance,
        totalMessages: messages.count || 0
      };
    }
  });

  const { data: recentActivity } = useQuery({
    queryKey: ["recent_activity"],
    queryFn: async () => {
      const { data } = await supabase
        .from("messages")
        .select("id, role, created_at")
        .order("created_at", { ascending: false })
        .limit(10);
      return data || [];
    }
  });

  const statCards = [
    {
      title: "Total Users",
      value: stats?.totalUsers || 0,
      icon: Users,
      color: "text-blue-500",
      bg: "bg-blue-500/10"
    },
    {
      title: "Active Subscriptions",
      value: stats?.activeSubscriptions || 0,
      icon: TrendingUp,
      color: "text-green-500",
      bg: "bg-green-500/10"
    },
    {
      title: "Total Wallet Balance",
      value: `৳${stats?.totalWalletBalance?.toFixed(2) || 0}`,
      icon: Wallet,
      color: "text-purple-500",
      bg: "bg-purple-500/10"
    },
    {
      title: "Total Messages",
      value: stats?.totalMessages || 0,
      icon: Activity,
      color: "text-orange-500",
      bg: "bg-orange-500/10"
    }
  ];

  return (
    <WalletGuard>
      <div className="min-h-screen bg-background p-6">
        <div className="max-w-7xl mx-auto">
          <div className="flex items-center gap-3 mb-6">
            <BarChart3 className="h-8 w-8 text-primary" />
            <div>
              <h1 className="text-3xl font-bold text-foreground">Analytics Dashboard</h1>
              <p className="text-muted-foreground">Smart Seba Platform Insights</p>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
            {statCards.map((stat) => {
              const Icon = stat.icon;
              return (
                <Card key={stat.title} className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-muted-foreground mb-1">{stat.title}</p>
                      <p className="text-2xl font-bold">{stat.value}</p>
                    </div>
                    <div className={`${stat.bg} p-3 rounded-lg`}>
                      <Icon className={`h-6 w-6 ${stat.color}`} />
                    </div>
                  </div>
                </Card>
              );
            })}
          </div>

          <Tabs defaultValue="overview" className="w-full">
            <TabsList>
              <TabsTrigger value="overview">Overview</TabsTrigger>
              <TabsTrigger value="activity">Recent Activity</TabsTrigger>
              <TabsTrigger value="modules">Top Modules</TabsTrigger>
            </TabsList>

            <TabsContent value="overview" className="mt-6">
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                <Card className="p-6">
                  <div className="flex items-center gap-2 mb-4">
                    <Calendar className="h-5 w-5 text-primary" />
                    <h3 className="text-lg font-semibold">Daily Active Users</h3>
                  </div>
                  <div className="text-3xl font-bold mb-2">
                    {Math.floor((stats?.totalUsers || 0) * 0.4)}
                  </div>
                  <p className="text-sm text-muted-foreground">40% of total users</p>
                </Card>

                <Card className="p-6">
                  <div className="flex items-center gap-2 mb-4">
                    <DollarSign className="h-5 w-5 text-primary" />
                    <h3 className="text-lg font-semibold">Revenue This Month</h3>
                  </div>
                  <div className="text-3xl font-bold mb-2">
                    ৳{((stats?.activeSubscriptions || 0) * 500).toFixed(2)}
                  </div>
                  <p className="text-sm text-muted-foreground">From subscriptions</p>
                </Card>
              </div>
            </TabsContent>

            <TabsContent value="activity" className="mt-6">
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4">Recent Platform Activity</h3>
                <div className="space-y-3">
                  {recentActivity?.map((activity) => (
                    <div key={activity.id} className="flex items-center justify-between py-2 border-b">
                      <div className="flex items-center gap-3">
                        <div className="w-2 h-2 rounded-full bg-primary" />
                        <span className="text-sm">
                          {activity.role === "user" ? "User message" : "AI response"}
                        </span>
                      </div>
                      <span className="text-xs text-muted-foreground">
                        {new Date(activity.created_at).toLocaleString()}
                      </span>
                    </div>
                  ))}
                </div>
              </Card>
            </TabsContent>

            <TabsContent value="modules" className="mt-6">
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4">Most Used Modules</h3>
                <div className="space-y-4">
                  {[
                    { name: "Smart Friend Chat", usage: "85%" },
                    { name: "Smart Academy", usage: "72%" },
                    { name: "Smart Pay", usage: "68%" },
                    { name: "Smart Bazar", usage: "54%" },
                    { name: "Smart Doctor", usage: "45%" }
                  ].map((module) => (
                    <div key={module.name}>
                      <div className="flex justify-between mb-1">
                        <span className="text-sm font-medium">{module.name}</span>
                        <span className="text-sm text-muted-foreground">{module.usage}</span>
                      </div>
                      <div className="w-full bg-muted rounded-full h-2">
                        <div 
                          className="bg-primary rounded-full h-2" 
                          style={{ width: module.usage }}
                        />
                      </div>
                    </div>
                  ))}
                </div>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </WalletGuard>
  );
};

export default StudioAnalytics;
