// Phase 3: Studio Builder Page

import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { useModuleBuilder } from '@/hooks/useModuleBuilder';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Textarea } from '@/components/ui/textarea';
import { BlueprintEditor } from '@/components/builder/BlueprintEditor';
import { ModuleBlueprint } from '@/types/moduleBlueprint';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { ArrowLeft, Sparkles, Download, CheckCircle } from 'lucide-react';
import { WalletGuard } from '@/components/ide/WalletGuard';
import { OwnerGuard } from '@/components/OwnerGuard';

const StudioBuilder = () => {
  const navigate = useNavigate();
  const { blueprints, createFromDescription, updateBlueprint, applyBlueprint } = useModuleBuilder();
  
  const [description, setDescription] = useState('');
  const [currentBlueprint, setCurrentBlueprint] = useState<ModuleBlueprint | null>(null);
  const [isGenerating, setIsGenerating] = useState(false);
  const [activeView, setActiveView] = useState<'create' | 'blueprints'>('create');

  const handleGenerate = async () => {
    if (!description.trim()) return;

    setIsGenerating(true);
    try {
      const result = await createFromDescription.mutateAsync({ description });
      setCurrentBlueprint(result.blueprint);
      setActiveView('blueprints');
    } catch (error) {
      console.error('Generation failed:', error);
    } finally {
      setIsGenerating(false);
    }
  };

  const handleSaveBlueprint = async () => {
    if (!currentBlueprint?.id) return;

    await updateBlueprint.mutateAsync({
      id: currentBlueprint.id,
      updates: currentBlueprint,
    });
  };

  const handleApplyBlueprint = async (blueprintId: string) => {
    await applyBlueprint.mutateAsync(blueprintId);
  };

  const downloadFiles = (files: Record<string, string>, moduleName: string) => {
    Object.entries(files).forEach(([path, content]) => {
      const blob = new Blob([content], { type: 'text/plain' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `${moduleName}-${path.replace(/\//g, '-')}`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    });
  };

  return (
    <WalletGuard>
      <OwnerGuard>
        <div className="h-screen bg-background flex flex-col">
          <div className="border-b border-border p-4 flex items-center justify-between">
            <div className="flex items-center gap-4">
              <Button
                variant="ghost"
                size="sm"
                onClick={() => navigate('/studio')}
                className="gap-2"
              >
                <ArrowLeft className="h-4 w-4" />
                Back to Studio
              </Button>
              <h1 className="text-xl font-semibold text-foreground">Module & Project Builder</h1>
            </div>
          </div>

          <div className="flex-1 overflow-hidden">
            <Tabs value={activeView} onValueChange={(v) => setActiveView(v as any)}>
              <div className="border-b border-border px-6">
                <TabsList>
                  <TabsTrigger value="create">Create New</TabsTrigger>
                  <TabsTrigger value="blueprints">My Blueprints ({blueprints.length})</TabsTrigger>
                </TabsList>
              </div>

              <div className="h-full overflow-auto p-6">
                <TabsContent value="create" className="mt-0">
                  <div className="max-w-4xl mx-auto space-y-6">
                    <Card>
                      <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                          <Sparkles className="h-5 w-5 text-primary" />
                          Describe Your Module or Project
                        </CardTitle>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <Textarea
                          placeholder="Example: Build a hospital management system with doctors, patients, appointments, and prescriptions. Include Laravel backend and React frontend."
                          value={description}
                          onChange={(e) => setDescription(e.target.value)}
                          rows={6}
                          className="resize-none"
                        />

                        <div className="flex gap-2">
                          <Button
                            onClick={handleGenerate}
                            disabled={!description.trim() || isGenerating}
                            className="flex-1"
                          >
                            {isGenerating ? (
                              <>Generating...</>
                            ) : (
                              <>
                                <Sparkles className="h-4 w-4 mr-2" />
                                Generate Module
                              </>
                            )}
                          </Button>
                        </div>
                      </CardContent>
                    </Card>

                    {currentBlueprint && (
                      <Card>
                        <CardHeader>
                          <CardTitle>Edit Blueprint</CardTitle>
                        </CardHeader>
                        <CardContent>
                          <BlueprintEditor
                            blueprint={currentBlueprint}
                            onChange={setCurrentBlueprint}
                            onSave={handleSaveBlueprint}
                          />
                        </CardContent>
                      </Card>
                    )}
                  </div>
                </TabsContent>

                <TabsContent value="blueprints" className="mt-0">
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    {blueprints.map((blueprint) => (
                      <Card key={blueprint.id}>
                        <CardHeader>
                          <CardTitle className="flex items-center justify-between">
                            {blueprint.name}
                            {blueprint.status === 'applied' && (
                              <CheckCircle className="h-4 w-4 text-green-500" />
                            )}
                          </CardTitle>
                          <p className="text-sm text-muted-foreground">{blueprint.description}</p>
                        </CardHeader>
                        <CardContent className="space-y-2">
                          <div className="text-xs text-muted-foreground space-y-1">
                            <div>Backend: {blueprint.backend_stack}</div>
                            <div>Frontend: {blueprint.frontend_stack}</div>
                            <div>Complexity: {blueprint.complexity}</div>
                            <div>Entities: {JSON.parse(blueprint.entities as any).length}</div>
                            <div>Status: <span className="font-medium">{blueprint.status}</span></div>
                          </div>

                          <div className="flex gap-2 pt-2">
                            {blueprint.status === 'generated' && (
                              <Button
                                size="sm"
                                onClick={() => handleApplyBlueprint(blueprint.id)}
                                className="flex-1"
                              >
                                <CheckCircle className="h-3 w-3 mr-1" />
                                Apply
                              </Button>
                            )}
                            {blueprint.generated_files && (
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => downloadFiles(
                                  blueprint.generated_files as any,
                                  blueprint.name
                                )}
                                className="flex-1"
                              >
                                <Download className="h-3 w-3 mr-1" />
                                Download
                              </Button>
                            )}
                          </div>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                </TabsContent>
              </div>
            </Tabs>
          </div>
        </div>
      </OwnerGuard>
    </WalletGuard>
  );
};

export default StudioBuilder;
