import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import {
  Phone,
  PhoneCall,
  PhoneOff,
  Mic,
  MicOff,
  Bot,
  History,
  Settings,
  Play,
  Square,
  MessageSquare,
  TrendingUp,
  Download
} from "lucide-react";
import { useAICallBot } from "@/hooks/useAICallBot";
import { useVoiceCall } from "@/hooks/useVoiceCall";
import { useCallHistory } from "@/hooks/useCallHistory";
import { WalletGuard } from "@/components/ide/WalletGuard";
import { cn } from "@/lib/utils";

const StudioCall = () => {
  const [activeTab, setActiveTab] = useState("bot");
  const [botConfig, setBotConfig] = useState({
    language: "en",
    voice: "alloy",
    systemPrompt: "You are a helpful AI assistant for Smart Friend Ultra.",
    enableSentiment: true,
    enableTicketing: true
  });

  const {
    startBot,
    stopBot,
    active: botActive,
    connecting: botConnecting,
    speaking,
    transcript,
    intent,
    sentiment
  } = useAICallBot();

  const {
    startCall,
    endCall,
    toggleMute,
    status: callStatus,
    peer,
    duration,
    muted
  } = useVoiceCall();

  const {
    history,
    loading: historyLoading,
    deleteRecord,
    exportTranscript
  } = useCallHistory();

  const formatDuration = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, "0")}`;
  };

  const handleStartBot = async () => {
    await startBot(botConfig);
  };

  return (
    <WalletGuard>
      <div className="h-screen bg-background flex flex-col">
        {/* Header */}
        <div className="border-b border-border p-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold">AI Call Center</h1>
              <p className="text-sm text-muted-foreground">
                Voice calling and AI-powered call bot system
              </p>
            </div>
            <Badge variant="outline">Phase 9</Badge>
          </div>
        </div>

        <div className="flex-1 overflow-auto p-6">
          <Tabs value={activeTab} onValueChange={setActiveTab}>
            <TabsList className="grid grid-cols-4 gap-2">
              <TabsTrigger value="bot">
                <Bot className="h-4 w-4 mr-2" />
                AI Call Bot
              </TabsTrigger>
              <TabsTrigger value="call">
                <Phone className="h-4 w-4 mr-2" />
                Voice Call
              </TabsTrigger>
              <TabsTrigger value="history">
                <History className="h-4 w-4 mr-2" />
                Call History
              </TabsTrigger>
              <TabsTrigger value="settings">
                <Settings className="h-4 w-4 mr-2" />
                Settings
              </TabsTrigger>
            </TabsList>

            {/* AI Call Bot Tab */}
            <TabsContent value="bot" className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                {/* Bot Control */}
                <Card>
                  <CardHeader>
                    <CardTitle>AI Call Bot Control</CardTitle>
                    <CardDescription>
                      Automated AI assistant for handling incoming calls
                    </CardDescription>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <div className="flex items-center justify-center p-8">
                      {!botActive ? (
                        <Button
                          onClick={handleStartBot}
                          disabled={botConnecting}
                          size="lg"
                          className="w-full"
                        >
                          {botConnecting ? (
                            <>
                              <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2" />
                              Connecting...
                            </>
                          ) : (
                            <>
                              <Play className="h-5 w-5 mr-2" />
                              Start AI Bot
                            </>
                          )}
                        </Button>
                      ) : (
                        <Button
                          onClick={stopBot}
                          variant="destructive"
                          size="lg"
                          className="w-full"
                        >
                          <Square className="h-5 w-5 mr-2" />
                          Stop AI Bot
                        </Button>
                      )}
                    </div>

                    <div className="grid grid-cols-2 gap-4 text-center">
                      <div>
                        <div className={cn(
                          "w-16 h-16 rounded-full mx-auto mb-2 flex items-center justify-center",
                          botActive ? "bg-green-500/20" : "bg-gray-500/20"
                        )}>
                          <Bot className={cn(
                            "h-8 w-8",
                            botActive ? "text-green-500" : "text-gray-500"
                          )} />
                        </div>
                        <p className="text-sm font-medium">
                          {botActive ? "Active" : "Inactive"}
                        </p>
                      </div>
                      <div>
                        <div className={cn(
                          "w-16 h-16 rounded-full mx-auto mb-2 flex items-center justify-center",
                          speaking ? "bg-blue-500/20 animate-pulse" : "bg-gray-500/20"
                        )}>
                          <Mic className={cn(
                            "h-8 w-8",
                            speaking ? "text-blue-500" : "text-gray-500"
                          )} />
                        </div>
                        <p className="text-sm font-medium">
                          {speaking ? "Speaking" : "Listening"}
                        </p>
                      </div>
                    </div>
                  </CardContent>
                </Card>

                {/* Live Analytics */}
                <Card>
                  <CardHeader>
                    <CardTitle>Live Analytics</CardTitle>
                    <CardDescription>Real-time call insights</CardDescription>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    {intent && (
                      <div className="space-y-2">
                        <Label>Detected Intent</Label>
                        <Badge variant="outline" className="w-full justify-start">
                          <MessageSquare className="h-4 w-4 mr-2" />
                          {intent.type} ({Math.round(intent.confidence * 100)}%)
                        </Badge>
                        <p className="text-sm text-muted-foreground">{intent.summary}</p>
                      </div>
                    )}

                    {sentiment && (
                      <div className="space-y-2">
                        <Label>Sentiment Analysis</Label>
                        <Badge 
                          variant="outline" 
                          className={cn(
                            "w-full justify-start",
                            sentiment.emotion === "positive" && "border-green-500",
                            sentiment.emotion === "negative" && "border-red-500",
                            sentiment.emotion === "frustrated" && "border-orange-500"
                          )}
                        >
                          <TrendingUp className="h-4 w-4 mr-2" />
                          {sentiment.emotion} ({Math.round(sentiment.score * 100)}%)
                        </Badge>
                      </div>
                    )}

                    {!intent && !sentiment && (
                      <div className="text-center py-8 text-muted-foreground">
                        Start the bot to see live analytics
                      </div>
                    )}
                  </CardContent>
                </Card>
              </div>

              {/* Live Transcript */}
              <Card>
                <CardHeader>
                  <CardTitle>Live Transcript</CardTitle>
                  <CardDescription>Real-time conversation transcription</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="h-64 overflow-auto space-y-2 bg-muted p-4 rounded-lg">
                    {transcript.map((entry, idx) => (
                      <div
                        key={idx}
                        className={cn(
                          "p-2 rounded",
                          entry.speaker === "bot" ? "bg-primary/10" : "bg-secondary/10"
                        )}
                      >
                        <span className="font-semibold text-xs">
                          {entry.speaker === "bot" ? "AI Bot" : "User"}:
                        </span>
                        <p className="text-sm">{entry.text}</p>
                        <span className="text-xs text-muted-foreground">
                          {new Date(entry.timestamp).toLocaleTimeString()}
                        </span>
                      </div>
                    ))}
                    {transcript.length === 0 && (
                      <div className="text-center py-12 text-muted-foreground">
                        Transcript will appear here during calls
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Voice Call Tab */}
            <TabsContent value="call" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>WebRTC Voice Calling</CardTitle>
                  <CardDescription>Browser-to-browser voice calls</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="flex items-center justify-center p-12">
                    {callStatus === "idle" && (
                      <div className="text-center space-y-4">
                        <div className="w-24 h-24 rounded-full bg-primary/10 mx-auto flex items-center justify-center">
                          <Phone className="h-12 w-12 text-primary" />
                        </div>
                        <p className="text-muted-foreground">Ready to make a call</p>
                        <Input placeholder="Enter user ID or phone" className="max-w-xs mx-auto" />
                        <Button onClick={() => startCall({ id: "test", name: "Test User" })}>
                          <PhoneCall className="h-4 w-4 mr-2" />
                          Start Call
                        </Button>
                      </div>
                    )}

                    {(callStatus === "calling" || callStatus === "ringing") && (
                      <div className="text-center space-y-4">
                        <div className="w-24 h-24 rounded-full bg-blue-500/20 mx-auto flex items-center justify-center animate-pulse">
                          <PhoneCall className="h-12 w-12 text-blue-500" />
                        </div>
                        <p className="font-semibold">{peer?.name}</p>
                        <p className="text-muted-foreground">
                          {callStatus === "calling" ? "Calling..." : "Ringing..."}
                        </p>
                        <Button onClick={endCall} variant="destructive">
                          <PhoneOff className="h-4 w-4 mr-2" />
                          Cancel
                        </Button>
                      </div>
                    )}

                    {callStatus === "connected" && (
                      <div className="text-center space-y-4">
                        <div className="w-24 h-24 rounded-full bg-green-500/20 mx-auto flex items-center justify-center">
                          <Phone className="h-12 w-12 text-green-500" />
                        </div>
                        <p className="font-semibold">{peer?.name}</p>
                        <p className="text-2xl font-mono">{formatDuration(duration)}</p>
                        <div className="flex gap-2 justify-center">
                          <Button onClick={toggleMute} variant="outline">
                            {muted ? <MicOff className="h-4 w-4" /> : <Mic className="h-4 w-4" />}
                          </Button>
                          <Button onClick={endCall} variant="destructive">
                            <PhoneOff className="h-4 w-4 mr-2" />
                            End Call
                          </Button>
                        </div>
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Call History Tab */}
            <TabsContent value="history" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>Call History</CardTitle>
                  <CardDescription>View and manage past calls</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-2">
                    {history.map((record) => (
                      <Card key={record.id}>
                        <CardContent className="p-4">
                          <div className="flex items-center justify-between">
                            <div className="space-y-1">
                              <p className="font-medium">
                                {new Date(record.created_at).toLocaleString()}
                              </p>
                              <p className="text-sm text-muted-foreground">
                                Duration: {formatDuration(record.call_duration)}
                              </p>
                              {record.ai_summary && (
                                <p className="text-sm">{record.ai_summary}</p>
                              )}
                            </div>
                            <div className="flex gap-2">
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => exportTranscript(record)}
                              >
                                <Download className="h-4 w-4" />
                              </Button>
                              <Button
                                size="sm"
                                variant="destructive"
                                onClick={() => deleteRecord(record.id)}
                              >
                                Delete
                              </Button>
                            </div>
                          </div>
                        </CardContent>
                      </Card>
                    ))}
                    {history.length === 0 && !historyLoading && (
                      <div className="text-center py-12 text-muted-foreground">
                        No call history yet
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Settings Tab */}
            <TabsContent value="settings" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>AI Bot Configuration</CardTitle>
                  <CardDescription>Configure the AI call bot behavior</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label>Language</Label>
                    <Select
                      value={botConfig.language}
                      onValueChange={(value) => setBotConfig({ ...botConfig, language: value })}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="en">English</SelectItem>
                        <SelectItem value="bn">Bangla</SelectItem>
                        <SelectItem value="es">Spanish</SelectItem>
                        <SelectItem value="fr">French</SelectItem>
                        <SelectItem value="de">German</SelectItem>
                        <SelectItem value="ja">Japanese</SelectItem>
                        <SelectItem value="zh">Chinese</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="space-y-2">
                    <Label>Voice</Label>
                    <Select
                      value={botConfig.voice}
                      onValueChange={(value) => setBotConfig({ ...botConfig, voice: value })}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="alloy">Alloy</SelectItem>
                        <SelectItem value="echo">Echo</SelectItem>
                        <SelectItem value="fable">Fable</SelectItem>
                        <SelectItem value="onyx">Onyx</SelectItem>
                        <SelectItem value="nova">Nova</SelectItem>
                        <SelectItem value="shimmer">Shimmer</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="space-y-2">
                    <Label>System Prompt</Label>
                    <Input
                      value={botConfig.systemPrompt}
                      onChange={(e) => setBotConfig({ ...botConfig, systemPrompt: e.target.value })}
                      placeholder="AI bot instructions"
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <Label>Enable Sentiment Analysis</Label>
                    <Switch
                      checked={botConfig.enableSentiment}
                      onCheckedChange={(checked) => setBotConfig({ ...botConfig, enableSentiment: checked })}
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <Label>Auto-Create Support Tickets</Label>
                    <Switch
                      checked={botConfig.enableTicketing}
                      onCheckedChange={(checked) => setBotConfig({ ...botConfig, enableTicketing: checked })}
                    />
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </WalletGuard>
  );
};

export default StudioCall;
