import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card } from "@/components/ui/card";
import { Switch } from "@/components/ui/switch";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Progress } from "@/components/ui/progress";
import { Badge } from "@/components/ui/badge";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { ScrollArea } from "@/components/ui/scroll-area";
import { useWebsiteCloner, CloneOptions } from "@/hooks/useWebsiteCloner";
import { 
  Copy, Download, Rocket, Shield, Smartphone, Search, 
  AlertTriangle, CheckCircle, FileCode, Folder 
} from "lucide-react";

const StudioClone = () => {
  const { cloneWebsite, downloadProject, resetCloner, cloning, progress, result } = useWebsiteCloner();
  
  const [options, setOptions] = useState<CloneOptions>({
    url: "",
    framework: "nextjs",
    useTailwind: true,
    enableSEO: true,
    enableResponsive: true,
    enableSecurity: true,
  });

  const [showWarning, setShowWarning] = useState(false);

  const handleClone = async () => {
    if (!options.url) return;
    
    // Show safety warning
    if (!showWarning) {
      setShowWarning(true);
      return;
    }

    await cloneWebsite(options);
  };

  const handleDownload = () => {
    if (result) {
      downloadProject(result);
    }
  };

  return (
    <div className="min-h-screen bg-background p-6">
      <div className="max-w-7xl mx-auto space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold flex items-center gap-2">
              <Copy className="w-8 h-8" />
              Website Cloner
            </h1>
            <p className="text-muted-foreground mt-1">
              Clone any website and generate a clean, production-ready project
            </p>
          </div>
        </div>

        {/* Safety Warning */}
        {showWarning && !result && (
          <Alert className="border-orange-500">
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              <strong>Important:</strong> This tool is for learning, redesigning, or rebuilding 
              your own sites or allowed templates only. Do not clone login-protected content, 
              copy trademarks, or reproduce copyrighted material.
            </AlertDescription>
          </Alert>
        )}

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Left Column - Configuration */}
          <div className="lg:col-span-1 space-y-6">
            <Card className="p-6">
              <h2 className="text-xl font-semibold mb-4">Clone Configuration</h2>
              
              <div className="space-y-4">
                {/* URL Input */}
                <div className="space-y-2">
                  <Label htmlFor="url">Target URL</Label>
                  <Input
                    id="url"
                    type="url"
                    placeholder="https://example.com"
                    value={options.url}
                    onChange={(e) => setOptions({ ...options, url: e.target.value })}
                  />
                </div>

                {/* Framework Selection */}
                <div className="space-y-2">
                  <Label>Framework</Label>
                  <Tabs 
                    value={options.framework} 
                    onValueChange={(v) => setOptions({ ...options, framework: v as "nextjs" | "react-spa" })}
                  >
                    <TabsList className="grid w-full grid-cols-2">
                      <TabsTrigger value="nextjs">Next.js</TabsTrigger>
                      <TabsTrigger value="react-spa">React SPA</TabsTrigger>
                    </TabsList>
                  </Tabs>
                </div>

                {/* Options */}
                <div className="space-y-3 pt-2">
                  <div className="flex items-center justify-between">
                    <Label htmlFor="tailwind" className="flex items-center gap-2">
                      Use Tailwind CSS
                    </Label>
                    <Switch
                      id="tailwind"
                      checked={options.useTailwind}
                      onCheckedChange={(checked) => setOptions({ ...options, useTailwind: checked })}
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <Label htmlFor="seo" className="flex items-center gap-2">
                      <Search className="w-4 h-4" />
                      SEO Optimization
                    </Label>
                    <Switch
                      id="seo"
                      checked={options.enableSEO}
                      onCheckedChange={(checked) => setOptions({ ...options, enableSEO: checked })}
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <Label htmlFor="responsive" className="flex items-center gap-2">
                      <Smartphone className="w-4 h-4" />
                      Responsive Fixes
                    </Label>
                    <Switch
                      id="responsive"
                      checked={options.enableResponsive}
                      onCheckedChange={(checked) => setOptions({ ...options, enableResponsive: checked })}
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <Label htmlFor="security" className="flex items-center gap-2">
                      <Shield className="w-4 h-4" />
                      Security Cleanup
                    </Label>
                    <Switch
                      id="security"
                      checked={options.enableSecurity}
                      onCheckedChange={(checked) => setOptions({ ...options, enableSecurity: checked })}
                    />
                  </div>
                </div>

                {/* Clone Button */}
                <Button 
                  onClick={handleClone} 
                  disabled={cloning || !options.url}
                  className="w-full mt-4"
                  size="lg"
                >
                  {cloning ? "Cloning..." : showWarning ? "I Understand, Start Clone" : "Analyze & Clone"}
                </Button>
              </div>
            </Card>
          </div>

          {/* Right Column - Progress & Results */}
          <div className="lg:col-span-2 space-y-6">
            {/* Progress */}
            {(cloning || progress.step !== "idle") && (
              <Card className="p-6">
                <h2 className="text-xl font-semibold mb-4">Clone Progress</h2>
                
                <div className="space-y-4">
                  <div className="space-y-2">
                    <div className="flex justify-between text-sm">
                      <span>{progress.message}</span>
                      <span>{progress.progress}%</span>
                    </div>
                    <Progress value={progress.progress} />
                  </div>

                  <div className="flex gap-2">
                    <Badge variant={progress.step === "fetching" ? "default" : "secondary"}>
                      Fetching
                    </Badge>
                    <Badge variant={progress.step === "analyzing" ? "default" : "secondary"}>
                      Analyzing
                    </Badge>
                    <Badge variant={progress.step === "building" ? "default" : "secondary"}>
                      Building
                    </Badge>
                    <Badge variant={progress.step === "generating" ? "default" : "secondary"}>
                      Generating
                    </Badge>
                    <Badge variant={progress.step === "complete" ? "default" : "secondary"}>
                      Complete
                    </Badge>
                  </div>
                </div>
              </Card>
            )}

            {/* Results */}
            {result && (
              <Card className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <h2 className="text-xl font-semibold flex items-center gap-2">
                    <CheckCircle className="w-5 h-5 text-green-500" />
                    Clone Complete
                  </h2>
                  <div className="flex gap-2">
                    <Button onClick={handleDownload} variant="outline">
                      <Download className="w-4 h-4 mr-2" />
                      Download ZIP
                    </Button>
                    <Button>
                      <Rocket className="w-4 h-4 mr-2" />
                      Deploy
                    </Button>
                  </div>
                </div>

                <Tabs defaultValue="summary">
                  <TabsList>
                    <TabsTrigger value="summary">Summary</TabsTrigger>
                    <TabsTrigger value="files">Files</TabsTrigger>
                    <TabsTrigger value="components">Components</TabsTrigger>
                  </TabsList>

                  <TabsContent value="summary" className="space-y-4">
                    <div className="grid grid-cols-3 gap-4">
                      <Card className="p-4">
                        <div className="text-2xl font-bold">{result.summary.totalFiles}</div>
                        <div className="text-sm text-muted-foreground">Files Generated</div>
                      </Card>
                      <Card className="p-4">
                        <div className="text-2xl font-bold">{result.summary.totalComponents}</div>
                        <div className="text-sm text-muted-foreground">Components</div>
                      </Card>
                      <Card className="p-4">
                        <div className="text-2xl font-bold">{result.summary.totalAssets}</div>
                        <div className="text-sm text-muted-foreground">Assets</div>
                      </Card>
                    </div>

                    <div>
                      <h3 className="font-semibold mb-2">Improvements Applied:</h3>
                      <ul className="space-y-1">
                        {result.summary.improvements.map((imp, idx) => (
                          <li key={idx} className="flex items-center gap-2 text-sm">
                            <CheckCircle className="w-4 h-4 text-green-500" />
                            {imp}
                          </li>
                        ))}
                      </ul>
                    </div>
                  </TabsContent>

                  <TabsContent value="files">
                    <ScrollArea className="h-[400px]">
                      <div className="space-y-2">
                        {Object.keys(result.files).map((file) => (
                          <div key={file} className="flex items-center gap-2 p-2 hover:bg-accent rounded">
                            <FileCode className="w-4 h-4" />
                            <span className="text-sm font-mono">{file}</span>
                          </div>
                        ))}
                      </div>
                    </ScrollArea>
                  </TabsContent>

                  <TabsContent value="components">
                    <ScrollArea className="h-[400px]">
                      <div className="grid grid-cols-2 gap-4">
                        {result.components.map((comp) => (
                          <Card key={comp} className="p-4">
                            <Folder className="w-8 h-8 mb-2 text-primary" />
                            <div className="font-semibold">{comp}</div>
                            <div className="text-xs text-muted-foreground">React Component</div>
                          </Card>
                        ))}
                      </div>
                    </ScrollArea>
                  </TabsContent>
                </Tabs>
              </Card>
            )}

            {/* Instructions */}
            {!result && !cloning && (
              <Card className="p-6">
                <h2 className="text-xl font-semibold mb-4">How to Use</h2>
                <ol className="space-y-2 text-sm">
                  <li>1. Enter the URL of the website you want to clone</li>
                  <li>2. Choose your preferred framework (Next.js or React SPA)</li>
                  <li>3. Enable improvements (SEO, Responsive, Security)</li>
                  <li>4. Click "Analyze & Clone" and accept the terms</li>
                  <li>5. Wait for the cloning process to complete</li>
                  <li>6. Download the generated project or deploy it directly</li>
                </ol>
              </Card>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default StudioClone;
