import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { useCommandParser } from "@/hooks/useCommandParser";
import { useCommandExecutor } from "@/hooks/useCommandExecutor";
import { useAgentManager } from "@/hooks/useAgentManager";
import { 
  Terminal, 
  Send, 
  Bot, 
  Clock, 
  CheckCircle2, 
  XCircle,
  Play,
  AlertCircle,
  Shield,
  Zap
} from "lucide-react";

export default function StudioCommand() {
  const [input, setInput] = useState("");
  const { execute: parseAndExecute } = useCommandParser();
  const { execute: executeCommand, executions, currentExecution } = useCommandExecutor();
  const { agents, agentStates, agentTasks, getStatusSummary } = useAgentManager();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!input.trim()) return;

    const parsed = parseAndExecute(input);
    if (parsed) {
      await executeCommand(parsed);
    }
    
    setInput("");
  };

  const summary = getStatusSummary();

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "idle": return <Clock className="h-4 w-4" />;
      case "busy": return <Play className="h-4 w-4 text-primary" />;
      case "error": return <XCircle className="h-4 w-4 text-destructive" />;
      case "disabled": return <AlertCircle className="h-4 w-4 text-muted-foreground" />;
      default: return <Clock className="h-4 w-4" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "idle": return "secondary";
      case "busy": return "default";
      case "error": return "destructive";
      case "disabled": return "outline";
      default: return "secondary";
    }
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center gap-3">
        <Terminal className="h-8 w-8 text-primary" />
        <div>
          <h1 className="text-3xl font-bold">Smart Friend-1 Command Center</h1>
          <p className="text-muted-foreground">Owner's Personal Boss Assistant</p>
        </div>
      </div>

      {/* Status Overview */}
      <div className="grid gap-4 md:grid-cols-5">
        <Card>
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium">Total Agents</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{summary.total}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium">Idle</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-muted-foreground">{summary.idle}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium">Busy</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-primary">{summary.busy}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium">Errors</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-destructive">{summary.error}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium">Disabled</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-muted-foreground">{summary.disabled}</div>
          </CardContent>
        </Card>
      </div>

      {/* Command Input */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Zap className="h-5 w-5" />
            Command Interface
          </CardTitle>
          <CardDescription>
            Type your command to execute tasks across 25 AI agents
          </CardDescription>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="flex gap-2">
            <Input
              value={input}
              onChange={(e) => setInput(e.target.value)}
              placeholder='Try: "build module Smart Academy" or "create ad: Spring Sale" or "fix everything"'
              className="flex-1"
            />
            <Button type="submit" disabled={!input.trim()}>
              <Send className="h-4 w-4 mr-2" />
              Execute
            </Button>
          </form>
        </CardContent>
      </Card>

      <div className="grid gap-6 md:grid-cols-2">
        {/* Current Execution */}
        {currentExecution && (
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Play className="h-5 w-5 text-primary" />
                Current Execution
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <div className="font-medium">{currentExecution.command.action}</div>
                  <div className="text-sm text-muted-foreground">
                    Agents: {currentExecution.command.agents.join(", ")}
                  </div>
                </div>
                
                <Badge variant={
                  currentExecution.status === "completed" ? "default" :
                  currentExecution.status === "failed" ? "destructive" :
                  currentExecution.status === "running" ? "default" : "secondary"
                }>
                  {currentExecution.status === "completed" && <CheckCircle2 className="h-3 w-3 mr-1" />}
                  {currentExecution.status === "failed" && <XCircle className="h-3 w-3 mr-1" />}
                  {currentExecution.status}
                </Badge>

                <ScrollArea className="h-32 rounded-md border p-3">
                  <div className="space-y-1 font-mono text-xs">
                    {currentExecution.logs.map((log, i) => (
                      <div key={i} className="text-muted-foreground">{log}</div>
                    ))}
                  </div>
                </ScrollArea>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Agent Status */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Bot className="h-5 w-5" />
              Agent Status
            </CardTitle>
            <CardDescription>Real-time agent activity</CardDescription>
          </CardHeader>
          <CardContent>
            <ScrollArea className="h-80">
              <div className="space-y-2">
                {/* SmartFriend 1 & 2 */}
                <div>
                  <div className="text-sm font-medium mb-2 flex items-center gap-2">
                    <Shield className="h-4 w-4 text-primary" />
                    Boss Agents (Level 1-2)
                  </div>
                  {agents.slice(0, 2).map(agent => (
                    <div key={agent.id} className="flex items-center justify-between py-2 px-3 rounded-md border mb-1">
                      <div className="flex items-center gap-2">
                        {getStatusIcon(agentStates[agent.id])}
                        <span className="text-sm font-medium">{agent.name}</span>
                      </div>
                      <Badge variant={getStatusColor(agentStates[agent.id]) as any}>
                        {agentStates[agent.id]}
                      </Badge>
                    </div>
                  ))}
                </div>

                <Separator className="my-3" />

                {/* Worker Agents */}
                <div>
                  <div className="text-sm font-medium mb-2 flex items-center gap-2">
                    <Bot className="h-4 w-4" />
                    Worker Agents (Level 3-25)
                  </div>
                  {agents.slice(2).map(agent => (
                    <div key={agent.id} className="flex items-center justify-between py-1.5 px-2 rounded-md hover:bg-accent/50 mb-1">
                      <div className="flex items-center gap-2">
                        {getStatusIcon(agentStates[agent.id])}
                        <span className="text-xs">{agent.name}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        {agentTasks[agent.id] && (
                          <span className="text-xs text-muted-foreground truncate max-w-[100px]">
                            {agentTasks[agent.id]}
                          </span>
                        )}
                        <Badge variant={getStatusColor(agentStates[agent.id]) as any} className="text-xs">
                          Lv{agent.level}
                        </Badge>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </ScrollArea>
          </CardContent>
        </Card>
      </div>

      {/* Command History */}
      <Card>
        <CardHeader>
          <CardTitle>Command History</CardTitle>
          <CardDescription>Recent command executions</CardDescription>
        </CardHeader>
        <CardContent>
          <ScrollArea className="h-64">
            <div className="space-y-2">
              {executions.length === 0 ? (
                <div className="text-center text-muted-foreground py-8">
                  No commands executed yet
                </div>
              ) : (
                executions.map((exec) => (
                  <div key={exec.id} className="border rounded-md p-3 space-y-2">
                    <div className="flex items-center justify-between">
                      <div className="font-medium">{exec.command.action}</div>
                      <Badge variant={
                        exec.status === "completed" ? "default" :
                        exec.status === "failed" ? "destructive" :
                        "secondary"
                      }>
                        {exec.status === "completed" && <CheckCircle2 className="h-3 w-3 mr-1" />}
                        {exec.status === "failed" && <XCircle className="h-3 w-3 mr-1" />}
                        {exec.status}
                      </Badge>
                    </div>
                    <div className="text-xs text-muted-foreground">
                      Agents: {exec.command.agents.join(", ")}
                    </div>
                    <div className="text-xs text-muted-foreground">
                      {exec.startTime.toLocaleString()}
                      {exec.endTime && ` → ${exec.endTime.toLocaleString()}`}
                    </div>
                  </div>
                ))
              )}
            </div>
          </ScrollArea>
        </CardContent>
      </Card>

      {/* Command Examples */}
      <Card>
        <CardHeader>
          <CardTitle>Command Examples</CardTitle>
          <CardDescription>Quick reference for available commands</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
            <div>
              <h4 className="font-medium mb-2">Global Commands</h4>
              <div className="text-sm space-y-1 text-muted-foreground">
                <div>• execute &lt;task&gt;</div>
                <div>• summary</div>
                <div>• status report</div>
                <div>• fix everything</div>
                <div>• optimize system</div>
              </div>
            </div>
            <div>
              <h4 className="font-medium mb-2">Module Building</h4>
              <div className="text-sm space-y-1 text-muted-foreground">
                <div>• build module &lt;name&gt;</div>
                <div>• update module &lt;name&gt;</div>
                <div>• fix module &lt;name&gt;</div>
              </div>
            </div>
            <div>
              <h4 className="font-medium mb-2">Development</h4>
              <div className="text-sm space-y-1 text-muted-foreground">
                <div>• generate backend for &lt;feature&gt;</div>
                <div>• generate frontend for &lt;feature&gt;</div>
                <div>• create api for &lt;name&gt;</div>
                <div>• review code</div>
              </div>
            </div>
            <div>
              <h4 className="font-medium mb-2">Marketing</h4>
              <div className="text-sm space-y-1 text-muted-foreground">
                <div>• write content: &lt;topic&gt;</div>
                <div>• create ad: &lt;product&gt;</div>
                <div>• create 30 social posts</div>
                <div>• design banner: &lt;details&gt;</div>
              </div>
            </div>
            <div>
              <h4 className="font-medium mb-2">Media</h4>
              <div className="text-sm space-y-1 text-muted-foreground">
                <div>• make pdf: &lt;topic&gt;</div>
                <div>• edit photo: &lt;describe&gt;</div>
                <div>• create promo video</div>
                <div>• enhance image quality</div>
              </div>
            </div>
            <div>
              <h4 className="font-medium mb-2">Child Mode</h4>
              <div className="text-sm space-y-1 text-muted-foreground">
                <div>• for Ava: &lt;task&gt;</div>
                <div>• send to Ava assistant: &lt;task&gt;</div>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
