import { useState } from "react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import {
  Upload,
  Server,
  Key,
  Globe,
  Shield,
  FileArchive,
  RotateCcw,
  CheckCircle,
  XCircle,
  Loader2,
  Terminal,
  Zap
} from "lucide-react";
import { useDeployment, DeploymentConfig } from "@/hooks/useDeployment";
import { useFTPDeploy } from "@/hooks/useFTPDeploy";
import { useCPanelDeploy } from "@/hooks/useCPanelDeploy";
import { useSSHDeploy } from "@/hooks/useSSHDeploy";
import { useDomainSetup } from "@/hooks/useDomainSetup";
import { useEnvironmentManager } from "@/hooks/useEnvironmentManager";
import { useRollback } from "@/hooks/useRollback";
import { WalletGuard } from "@/components/ide/WalletGuard";

const StudioDeploy = () => {
  const [activeTab, setActiveTab] = useState("ftp");
  const [config, setConfig] = useState<Partial<DeploymentConfig>>({
    method: "ftp",
    host: "",
    port: 21,
    username: "",
    password: "",
    path: "/public_html",
    domain: "",
    autoSSL: true
  });

  const { deploy, rollback, deploying, currentStatus, logs, deployments } = useDeployment();
  const { testConnection, connected: ftpConnected } = useFTPDeploy();
  const { validateCredentials } = useCPanelDeploy();
  const { connect, connected: sshConnected } = useSSHDeploy();
  const { configureDomain, dnsRecords } = useDomainSetup();
  const { generateEnvFile, detectVariables } = useEnvironmentManager();
  const { createRollbackPoint, rollbackTo, rollbackPoints, rolling } = useRollback();

  const handleDeploy = async () => {
    if (!config.host || !config.username) {
      return;
    }
    
    await createRollbackPoint("Before deployment");
    await deploy(config as DeploymentConfig);
  };

  const handleSmartSebaDeployment = async () => {
    const smartSebaConfig: DeploymentConfig = {
      method: "smartseba",
      host: "smartseba.com",
      username: "auto",
      path: "/apps/smartfriend",
      domain: config.domain || "app.smartseba.com",
      autoSSL: true
    };

    await createRollbackPoint("Before SmartSeba deployment");
    await deploy(smartSebaConfig);
  };

  return (
    <WalletGuard>
      <div className="h-screen bg-background flex flex-col">
        {/* Header */}
        <div className="border-b border-border p-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold">Self Deploy Engine</h1>
              <p className="text-sm text-muted-foreground">
                Deploy to any server with FTP, cPanel, SSH, or SmartSeba Hosting
              </p>
            </div>
            <Badge variant="outline">Phase 8</Badge>
          </div>
        </div>

        <div className="flex-1 overflow-auto p-6">
          <Tabs value={activeTab} onValueChange={setActiveTab}>
            <TabsList className="grid grid-cols-7 gap-2">
              <TabsTrigger value="ftp">FTP</TabsTrigger>
              <TabsTrigger value="cpanel">cPanel</TabsTrigger>
              <TabsTrigger value="ssh">SSH/VPS</TabsTrigger>
              <TabsTrigger value="smartseba">SmartSeba</TabsTrigger>
              <TabsTrigger value="domain">Domain</TabsTrigger>
              <TabsTrigger value="env">Environment</TabsTrigger>
              <TabsTrigger value="rollback">Rollback</TabsTrigger>
            </TabsList>

            {/* FTP Deployment */}
            <TabsContent value="ftp" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>FTP Deployment</CardTitle>
                  <CardDescription>Deploy via FTP to shared hosting</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label>FTP Host</Label>
                      <Input
                        placeholder="ftp.example.com"
                        value={config.host}
                        onChange={(e) => setConfig({ ...config, host: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Port</Label>
                      <Input
                        type="number"
                        placeholder="21"
                        value={config.port}
                        onChange={(e) => setConfig({ ...config, port: parseInt(e.target.value) })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Username</Label>
                      <Input
                        placeholder="ftpuser"
                        value={config.username}
                        onChange={(e) => setConfig({ ...config, username: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Password</Label>
                      <Input
                        type="password"
                        placeholder="••••••••"
                        value={config.password}
                        onChange={(e) => setConfig({ ...config, password: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Remote Path</Label>
                      <Input
                        placeholder="/public_html"
                        value={config.path}
                        onChange={(e) => setConfig({ ...config, path: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Domain</Label>
                      <Input
                        placeholder="example.com"
                        value={config.domain}
                        onChange={(e) => setConfig({ ...config, domain: e.target.value })}
                      />
                    </div>
                  </div>

                  <div className="flex items-center space-x-2">
                    <Switch
                      checked={config.autoSSL}
                      onCheckedChange={(checked) => setConfig({ ...config, autoSSL: checked })}
                    />
                    <Label>Auto-install SSL (Let's Encrypt)</Label>
                  </div>

                  <div className="flex gap-2">
                    <Button
                      onClick={() => testConnection({
                        host: config.host!,
                        port: config.port!,
                        username: config.username!,
                        password: config.password!,
                        path: config.path!
                      })}
                    >
                      <Server className="h-4 w-4 mr-2" />
                      Test Connection
                    </Button>
                    <Button onClick={handleDeploy} disabled={deploying}>
                      {deploying ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Upload className="h-4 w-4 mr-2" />}
                      Deploy Now
                    </Button>
                  </div>

                  {ftpConnected && (
                    <Badge variant="default" className="w-fit">
                      <CheckCircle className="h-3 w-3 mr-1" />
                      Connected
                    </Badge>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            {/* cPanel Deployment */}
            <TabsContent value="cpanel" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>cPanel API Deployment</CardTitle>
                  <CardDescription>Deploy using cPanel API (recommended for shared hosting)</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label>cPanel Host</Label>
                      <Input
                        placeholder="cpanel.example.com"
                        value={config.host}
                        onChange={(e) => setConfig({ ...config, host: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Username</Label>
                      <Input
                        placeholder="cpanel_user"
                        value={config.username}
                        onChange={(e) => setConfig({ ...config, username: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2 col-span-2">
                      <Label>API Token</Label>
                      <Input
                        type="password"
                        placeholder="Your cPanel API token"
                        value={config.password}
                        onChange={(e) => setConfig({ ...config, password: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Domain</Label>
                      <Input
                        placeholder="example.com"
                        value={config.domain}
                        onChange={(e) => setConfig({ ...config, domain: e.target.value })}
                      />
                    </div>
                  </div>

                  <div className="flex gap-2">
                    <Button
                      onClick={() => validateCredentials({
                        host: config.host!,
                        username: config.username!,
                        apiToken: config.password!,
                        domain: config.domain!
                      })}
                    >
                      <Key className="h-4 w-4 mr-2" />
                      Validate Credentials
                    </Button>
                    <Button onClick={handleDeploy} disabled={deploying}>
                      {deploying ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Upload className="h-4 w-4 mr-2" />}
                      Deploy Now
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* SSH Deployment */}
            <TabsContent value="ssh" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>SSH/VPS Deployment</CardTitle>
                  <CardDescription>Deploy to VPS or dedicated server via SSH</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label>SSH Host</Label>
                      <Input
                        placeholder="server.example.com"
                        value={config.host}
                        onChange={(e) => setConfig({ ...config, host: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Port</Label>
                      <Input
                        type="number"
                        placeholder="22"
                        value={config.port || 22}
                        onChange={(e) => setConfig({ ...config, port: parseInt(e.target.value) })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Username</Label>
                      <Input
                        placeholder="root"
                        value={config.username}
                        onChange={(e) => setConfig({ ...config, username: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Authentication Method</Label>
                      <select className="w-full p-2 border rounded">
                        <option>Password</option>
                        <option>Private Key</option>
                      </select>
                    </div>
                    <div className="space-y-2 col-span-2">
                      <Label>Password / Private Key</Label>
                      <Textarea
                        placeholder="Enter password or paste private key"
                        value={config.password}
                        onChange={(e) => setConfig({ ...config, password: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label>Deployment Path</Label>
                      <Input
                        placeholder="/var/www/html"
                        value={config.path}
                        onChange={(e) => setConfig({ ...config, path: e.target.value })}
                      />
                    </div>
                  </div>

                  <div className="flex gap-2">
                    <Button
                      onClick={() => connect({
                        host: config.host!,
                        port: config.port || 22,
                        username: config.username!,
                        password: config.password
                      })}
                    >
                      <Terminal className="h-4 w-4 mr-2" />
                      Test SSH Connection
                    </Button>
                    <Button onClick={handleDeploy} disabled={deploying}>
                      {deploying ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Upload className="h-4 w-4 mr-2" />}
                      Deploy Now
                    </Button>
                  </div>

                  {sshConnected && (
                    <Badge variant="default" className="w-fit">
                      <CheckCircle className="h-3 w-3 mr-1" />
                      SSH Connected
                    </Badge>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            {/* SmartSeba One-Click */}
            <TabsContent value="smartseba" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>SmartSeba Hosting</CardTitle>
                  <CardDescription>One-click deployment to SmartSeba's optimized hosting</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="bg-gradient-to-r from-primary/10 to-secondary/10 p-6 rounded-lg space-y-4">
                    <div className="flex items-center gap-3">
                      <Zap className="h-8 w-8 text-primary" />
                      <div>
                        <h3 className="font-semibold text-lg">Instant Deployment</h3>
                        <p className="text-sm text-muted-foreground">
                          Deploy to SmartSeba's managed hosting with automatic SSL, CDN, and optimization
                        </p>
                      </div>
                    </div>

                    <div className="grid grid-cols-3 gap-4 text-center">
                      <div>
                        <Shield className="h-6 w-6 mx-auto mb-2 text-green-500" />
                        <p className="text-xs font-medium">Auto SSL</p>
                      </div>
                      <div>
                        <Globe className="h-6 w-6 mx-auto mb-2 text-blue-500" />
                        <p className="text-xs font-medium">Global CDN</p>
                      </div>
                      <div>
                        <Zap className="h-6 w-6 mx-auto mb-2 text-yellow-500" />
                        <p className="text-xs font-medium">Auto Scaling</p>
                      </div>
                    </div>
                  </div>

                  <div className="space-y-2">
                    <Label>Custom Domain (Optional)</Label>
                    <Input
                      placeholder="your-domain.com"
                      value={config.domain}
                      onChange={(e) => setConfig({ ...config, domain: e.target.value })}
                    />
                    <p className="text-xs text-muted-foreground">
                      Leave blank to use smartseba.com subdomain
                    </p>
                  </div>

                  <Button onClick={handleSmartSebaDeployment} disabled={deploying} className="w-full" size="lg">
                    {deploying ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Zap className="h-4 w-4 mr-2" />}
                    Deploy to SmartSeba Now
                  </Button>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Domain Setup */}
            <TabsContent value="domain" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>Domain & SSL Configuration</CardTitle>
                  <CardDescription>Automatically configure domain DNS and SSL certificate</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label>Domain Name</Label>
                    <Input placeholder="example.com" />
                  </div>

                  <div className="space-y-2">
                    <Label>Server IP Address</Label>
                    <Input placeholder="192.168.1.1" />
                  </div>

                  <Button
                    onClick={() => configureDomain({
                      domain: config.domain!,
                      ip: "192.168.1.1",
                      autoSSL: true
                    })}
                  >
                    <Globe className="h-4 w-4 mr-2" />
                    Configure Domain
                  </Button>

                  {dnsRecords.length > 0 && (
                    <div className="space-y-2">
                      <Label>DNS Records to Add</Label>
                      <Card>
                        <CardContent className="p-4 space-y-2">
                          {dnsRecords.map((record, idx) => (
                            <div key={idx} className="flex items-center justify-between text-sm">
                              <Badge variant="outline">{record.type}</Badge>
                              <span className="text-muted-foreground">{record.name}</span>
                              <span className="font-mono">{record.value}</span>
                            </div>
                          ))}
                        </CardContent>
                      </Card>
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            {/* Environment Manager */}
            <TabsContent value="env" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>Environment Variables</CardTitle>
                  <CardDescription>Generate and manage .env files for deployment</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="flex gap-2">
                    <Button onClick={() => detectVariables()}>
                      <FileArchive className="h-4 w-4 mr-2" />
                      Auto-Detect Variables
                    </Button>
                    <Button onClick={() => generateEnvFile("production", [])}>
                      Generate .env File
                    </Button>
                  </div>

                  <div className="space-y-2">
                    <Label>Environment Preview</Label>
                    <Textarea
                      className="font-mono text-xs"
                      rows={10}
                      placeholder="# Generated .env file will appear here"
                      readOnly
                    />
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Rollback System */}
            <TabsContent value="rollback" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>Rollback System</CardTitle>
                  <CardDescription>Restore previous deployments if something goes wrong</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <Button onClick={() => createRollbackPoint("Manual backup")}>
                    <FileArchive className="h-4 w-4 mr-2" />
                    Create Rollback Point
                  </Button>

                  <div className="space-y-2">
                    <Label>Available Rollback Points ({rollbackPoints.length})</Label>
                    {rollbackPoints.map((point) => (
                      <Card key={point.id}>
                        <CardContent className="p-4">
                          <div className="flex items-center justify-between">
                            <div>
                              <p className="font-medium">{point.description}</p>
                              <p className="text-xs text-muted-foreground">
                                {new Date(point.timestamp).toLocaleString()}
                              </p>
                            </div>
                            <Button size="sm" onClick={() => rollbackTo(point.id)} disabled={rolling}>
                              {rolling ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <RotateCcw className="h-4 w-4 mr-2" />}
                              Rollback
                            </Button>
                          </div>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>

          {/* Deployment Logs */}
          {logs.length > 0 && (
            <Card className="mt-6">
              <CardHeader>
                <CardTitle>Deployment Logs</CardTitle>
                <Badge variant="outline">{currentStatus}</Badge>
              </CardHeader>
              <CardContent>
                <div className="bg-black text-green-400 p-4 rounded-lg font-mono text-sm h-64 overflow-auto space-y-1">
                  {logs.map((log, idx) => (
                    <div key={idx} className="flex items-start gap-2">
                      {log.level === "error" && <XCircle className="h-4 w-4 text-red-500 flex-shrink-0" />}
                      {log.level === "success" && <CheckCircle className="h-4 w-4 text-green-500 flex-shrink-0" />}
                      {log.level === "warning" && <Shield className="h-4 w-4 text-yellow-500 flex-shrink-0" />}
                      {log.level === "info" && <Terminal className="h-4 w-4 text-blue-500 flex-shrink-0" />}
                      <span className="text-gray-400">[{new Date(log.timestamp).toLocaleTimeString()}]</span>
                      <span>{log.message}</span>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </WalletGuard>
  );
};

export default StudioDeploy;
