import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { WalletGuard } from "@/components/ide/WalletGuard";
import { Palette, Loader2, Sparkles, Eye, Download } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { supabase } from "@/integrations/supabase/client";

const StudioDesigner = () => {
  const [url, setUrl] = useState("");
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [analysis, setAnalysis] = useState<any>(null);
  const { toast } = useToast();

  const handleAnalyze = async () => {
    if (!url.trim()) {
      toast({ title: "Error", description: "Please enter a URL", variant: "destructive" });
      return;
    }

    setIsAnalyzing(true);
    try {
      const { data, error } = await supabase.functions.invoke("analyze-site", {
        body: { url }
      });

      if (error) throw error;

      setAnalysis(data);
      toast({ title: "Success", description: "Website analyzed successfully" });
    } catch (error) {
      console.error("Analysis error:", error);
      toast({ 
        title: "Error", 
        description: "Failed to analyze website", 
        variant: "destructive" 
      });
    } finally {
      setIsAnalyzing(false);
    }
  };

  return (
    <WalletGuard>
      <div className="h-screen bg-gradient-to-br from-primary/5 via-background to-secondary/5 p-6 overflow-y-auto">
        <div className="max-w-6xl mx-auto space-y-6">
          <div className="flex items-center gap-3">
            <div className="p-3 bg-gradient-to-br from-primary/20 to-secondary/20 rounded-xl">
              <Palette className="h-8 w-8 text-primary" />
            </div>
            <div>
              <h1 className="text-3xl font-bold bg-gradient-to-r from-primary to-secondary bg-clip-text text-transparent">
                Smart Website Designer
              </h1>
              <p className="text-muted-foreground">Analyze and improve website designs with AI</p>
            </div>
          </div>

          <Card className="shadow-xl">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Sparkles className="h-5 w-5" />
                Website Analyzer
              </CardTitle>
              <CardDescription>
                Enter a URL to analyze its design and generate improvements
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex gap-2">
                <Input
                  placeholder="https://example.com"
                  value={url}
                  onChange={(e) => setUrl(e.target.value)}
                  className="flex-1"
                />
                <Button 
                  onClick={handleAnalyze} 
                  disabled={isAnalyzing || !url.trim()}
                  className="gap-2"
                >
                  {isAnalyzing ? (
                    <>
                      <Loader2 className="h-4 w-4 animate-spin" />
                      Analyzing...
                    </>
                  ) : (
                    <>
                      <Eye className="h-4 w-4" />
                      Analyze
                    </>
                  )}
                </Button>
              </div>

              {analysis && (
                <div className="mt-6 space-y-4">
                  <div className="p-4 bg-gradient-to-r from-primary/10 to-secondary/10 rounded-lg">
                    <h3 className="font-semibold mb-2">Analysis Results</h3>
                    <div className="space-y-2 text-sm">
                      <p><strong>Colors:</strong> {analysis.colors?.join(", ") || "N/A"}</p>
                      <p><strong>Fonts:</strong> {analysis.fonts?.join(", ") || "N/A"}</p>
                      <p><strong>Layout:</strong> {analysis.layout || "N/A"}</p>
                    </div>
                  </div>

                  <div className="flex gap-2">
                    <Button className="flex-1 gap-2">
                      <Sparkles className="h-4 w-4" />
                      Generate Improved Design
                    </Button>
                    <Button variant="outline" className="gap-2">
                      <Download className="h-4 w-4" />
                      Export
                    </Button>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>

          <Card className="shadow-xl">
            <CardHeader>
              <CardTitle>Design Preview</CardTitle>
              <CardDescription>
                Preview your generated design in real-time
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="aspect-video bg-muted rounded-lg flex items-center justify-center">
                <p className="text-muted-foreground">
                  {analysis ? "Design preview will appear here" : "Analyze a website to see preview"}
                </p>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </WalletGuard>
  );
};

export default StudioDesigner;
