import { useState } from "react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { 
  Bug, 
  Wrench, 
  Rocket, 
  Code, 
  Container, 
  Server, 
  Github, 
  GitBranch,
  Play,
  CheckCircle,
  XCircle,
  Loader2
} from "lucide-react";
import { useErrorScanner } from "@/hooks/useErrorScanner";
import { useAutoFix } from "@/hooks/useAutoFix";
import { useLaravelGenerator } from "@/hooks/useLaravelGenerator";
import { useBackendFixer } from "@/hooks/useBackendFixer";
import { useFrontendBuilder } from "@/hooks/useFrontendBuilder";
import { useDockerGenerator } from "@/hooks/useDockerGenerator";
import { useServerTester } from "@/hooks/useServerTester";
import { useGitHubIntegration } from "@/hooks/useGitHubIntegration";
import { useCICDGenerator } from "@/hooks/useCICDGenerator";
import { useComposerSetup } from "@/hooks/useComposerSetup";
import { useLaravelPostDeploy } from "@/hooks/useLaravelPostDeploy";
import { getLaravelConfig, saveLaravelConfig, LaravelRuntimeConfig } from "@/lib/devops/laravelRuntimeConfig";
import { Checkbox } from "@/components/ui/checkbox";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Label } from "@/components/ui/label";
import { toast } from "sonner";

const StudioDevOps = () => {
  const [activeTab, setActiveTab] = useState("scanner");
  const [laravelConfig, setLaravelConfig] = useState<LaravelRuntimeConfig>(getLaravelConfig());
  
  // Error Detection & Auto-Fix
  const { scanProject, errors, isScanning } = useErrorScanner();
  const { generateFix, applyPatch, patches, generating } = useAutoFix();
  
  // Backend Generation
  const { generateAPI, exportBackend, projects: laravelProjects, generating: genLaravel } = useLaravelGenerator();
  const { lintBackend, fixMigrations, runComposerDiagnose, fixComposer, fixing } = useBackendFixer();
  
  // Frontend Building
  const { buildReact, buildNext, buildStatus, building } = useFrontendBuilder();
  
  // Docker & Deployment
  const { generateDockerfile, generateDockerCompose, generating: genDocker } = useDockerGenerator();
  
  // Server Testing
  const { testPHP, testMySQL, testSSL, testDomain, testResults, testing } = useServerTester();
  
  // GitHub & CI/CD
  const { pushToGitHub, createPR, pushing } = useGitHubIntegration();
  const { generateGitHubActions, generateGitLabCI, pipelines, generating: genPipeline } = useCICDGenerator();
  
  // Laravel Runtime
  const { runComposerSetup, runComposerUpdate, installing: installingComposer, logs: composerLogs } = useComposerSetup();
  const { runLaravelPostDeploy, runSingleArtisanCommand, running: runningArtisan, logs: artisanLogs, results: artisanResults } = useLaravelPostDeploy();

  const handleScanAndFix = async () => {
    await scanProject();
    toast.success("Project scanned");
  };

  const handleGenerateAPI = async () => {
    const config = {
      name: "BlogAPI",
      resource: "Post",
      fields: [
        { name: "title", type: "string" },
        { name: "content", type: "text" },
        { name: "published", type: "boolean" }
      ]
    };
    await generateAPI(config);
    toast.success("Laravel API generated");
  };

  const handleBuildFrontend = async () => {
    await buildReact();
    toast.success("Frontend build started");
  };

  const handleGenerateDocker = async () => {
    await generateDockerfile("node:18-alpine", ["npm install", "npm run build"]);
    await generateDockerCompose();
    toast.success("Docker files generated");
  };

  const handleRunServerTests = async () => {
    await testPHP();
    await testMySQL();
    await testSSL();
    await testDomain("example.com");
    toast.success("Server tests completed");
  };

  const handleGeneratePipeline = async () => {
    await generateGitHubActions({
      name: "Smart Friend Ultra CI/CD",
      triggers: ["push", "pull_request"],
      jobs: ["build", "test", "deploy"]
    });
    toast.success("CI/CD pipeline generated");
  };

  const handleRunComposer = async () => {
    const mockTarget = {
      type: "ssh" as const,
      host: "example.com",
      username: "user"
    };
    await runComposerSetup(mockTarget, laravelConfig);
  };

  const handleRunArtisan = async () => {
    const mockTarget = {
      type: "ssh" as const,
      host: "example.com",
      username: "user"
    };
    await runLaravelPostDeploy(mockTarget, laravelConfig);
  };

  const updateLaravelConfig = (updates: Partial<LaravelRuntimeConfig>) => {
    const updated = saveLaravelConfig(updates);
    setLaravelConfig(updated);
    toast.success("Laravel config updated");
  };

  return (
    <div className="h-screen bg-background flex flex-col">
      {/* Header */}
      <div className="border-b border-border p-4">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-2xl font-bold">Auto DevOps Engine</h1>
            <p className="text-sm text-muted-foreground">
              Automated development operations & deployment
            </p>
          </div>
          <Badge variant="outline" className="text-xs">
            Phase 7 - Active
          </Badge>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 overflow-auto p-6">
        <Tabs value={activeTab} onValueChange={setActiveTab}>
          <TabsList className="grid grid-cols-4 lg:grid-cols-10 gap-2">
            <TabsTrigger value="scanner" className="gap-2">
              <Bug className="h-4 w-4" />
              Scanner
            </TabsTrigger>
            <TabsTrigger value="fixer" className="gap-2">
              <Wrench className="h-4 w-4" />
              Auto-Fix
            </TabsTrigger>
            <TabsTrigger value="backend" className="gap-2">
              <Code className="h-4 w-4" />
              Backend
            </TabsTrigger>
            <TabsTrigger value="frontend" className="gap-2">
              <Rocket className="h-4 w-4" />
              Frontend
            </TabsTrigger>
            <TabsTrigger value="docker" className="gap-2">
              <Container className="h-4 w-4" />
              Docker
            </TabsTrigger>
            <TabsTrigger value="server" className="gap-2">
              <Server className="h-4 w-4" />
              Server Test
            </TabsTrigger>
            <TabsTrigger value="github" className="gap-2">
              <Github className="h-4 w-4" />
              GitHub
            </TabsTrigger>
            <TabsTrigger value="cicd" className="gap-2">
              <GitBranch className="h-4 w-4" />
              CI/CD
            </TabsTrigger>
            <TabsTrigger value="laravel" className="gap-2">
              <Code className="h-4 w-4" />
              Laravel
            </TabsTrigger>
            <TabsTrigger value="console" className="gap-2">
              <Play className="h-4 w-4" />
              Console
            </TabsTrigger>
          </TabsList>

          {/* Error Scanner Tab */}
          <TabsContent value="scanner" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Error Detection & Analysis</CardTitle>
                <CardDescription>Scan project for build errors and code issues</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <Button onClick={handleScanAndFix} disabled={isScanning}>
                  {isScanning ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Bug className="h-4 w-4 mr-2" />}
                  {isScanning ? "Scanning..." : "Scan Project"}
                </Button>

                <div className="space-y-2">
                  <h4 className="font-semibold">Detected Errors ({errors.length})</h4>
                  {errors.map((error, idx) => (
                    <Card key={idx} className="border-l-4 border-l-red-500">
                      <CardContent className="p-4">
                        <div className="flex items-start justify-between">
                          <div className="space-y-1">
                            <Badge variant={error.severity === "critical" ? "destructive" : "outline"}>
                              {error.severity}
                            </Badge>
                            <p className="font-medium">{error.type}</p>
                            <p className="text-sm text-muted-foreground">{error.message}</p>
                            <p className="text-xs text-muted-foreground">
                              {error.file}:{error.line}
                            </p>
                          </div>
                          <Button size="sm" onClick={() => generateFix(error)} disabled={generating}>
                            Generate Fix
                          </Button>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Auto-Fix Tab */}
          <TabsContent value="fixer" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Auto-Fix Patches</CardTitle>
                <CardDescription>Review and apply generated code patches</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                {patches.map((patch) => (
                  <Card key={patch.id}>
                    <CardContent className="p-4">
                      <div className="space-y-2">
                        <div className="flex items-center justify-between">
                          <Badge variant={patch.applied ? "default" : "outline"}>
                            {patch.applied ? "Applied" : "Pending"}
                          </Badge>
                          <Button size="sm" onClick={() => applyPatch(patch.id)} disabled={patch.applied}>
                            {patch.applied ? <CheckCircle className="h-4 w-4 mr-2" /> : <Play className="h-4 w-4 mr-2" />}
                            {patch.applied ? "Applied" : "Apply Patch"}
                          </Button>
                        </div>
                        <p className="font-medium">{patch.file}</p>
                        <p className="text-sm text-muted-foreground">{patch.description}</p>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Backend Generator Tab */}
          <TabsContent value="backend" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Laravel Backend Generator</CardTitle>
                <CardDescription>Generate complete Laravel API with routes, controllers, models, and migrations</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <Button onClick={handleGenerateAPI} disabled={genLaravel}>
                    {genLaravel ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Code className="h-4 w-4 mr-2" />}
                    Generate API
                  </Button>
                  <Button onClick={() => lintBackend()} disabled={fixing}>
                    Auto-Lint Backend
                  </Button>
                  <Button onClick={() => fixMigrations()} disabled={fixing}>
                    Fix Migrations
                  </Button>
                  <Button onClick={() => runComposerDiagnose()} disabled={fixing}>
                    Composer Diagnose
                  </Button>
                </div>

                <div className="space-y-2">
                  <h4 className="font-semibold">Generated Projects ({laravelProjects.length})</h4>
                  {laravelProjects.map((project) => (
                    <Card key={project.name}>
                      <CardContent className="p-4">
                        <div className="flex items-center justify-between">
                          <div>
                            <p className="font-medium">{project.name}</p>
                            <p className="text-xs text-muted-foreground">
                              {project.components.length} files • {project.created}
                            </p>
                          </div>
                          <Button size="sm" onClick={() => exportBackend(project.name)}>
                            Export ZIP
                          </Button>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Frontend Builder Tab */}
          <TabsContent value="frontend" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Frontend Builder</CardTitle>
                <CardDescription>Build React/Next.js applications</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <Button onClick={handleBuildFrontend} disabled={building}>
                    {building ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Rocket className="h-4 w-4 mr-2" />}
                    Build React App
                  </Button>
                  <Button onClick={() => buildNext()} disabled={building}>
                    Build Next.js App
                  </Button>
                </div>

                {buildStatus && (
                  <Card>
                    <CardContent className="p-4">
                      <div className="space-y-2">
                        <Badge variant={buildStatus.success ? "default" : "destructive"}>
                          {buildStatus.success ? "Success" : "Failed"}
                        </Badge>
                        <p className="text-sm">{buildStatus.message}</p>
                        {buildStatus.duration && (
                          <p className="text-xs text-muted-foreground">
                            Build time: {buildStatus.duration}ms
                          </p>
                        )}
                      </div>
                    </CardContent>
                  </Card>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Docker Tab */}
          <TabsContent value="docker" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Docker Generator</CardTitle>
                <CardDescription>Generate Dockerfile and docker-compose.yml automatically</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <Button onClick={handleGenerateDocker} disabled={genDocker}>
                  {genDocker ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Container className="h-4 w-4 mr-2" />}
                  Generate Docker Files
                </Button>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Server Tester Tab */}
          <TabsContent value="server" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Server Tester</CardTitle>
                <CardDescription>Test PHP, MySQL, SSL, and domain configuration</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <Button onClick={handleRunServerTests} disabled={testing}>
                  {testing ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Server className="h-4 w-4 mr-2" />}
                  Run All Tests
                </Button>

                <div className="space-y-2">
                  {testResults.map((result) => (
                    <Card key={result.test}>
                      <CardContent className="p-4">
                        <div className="flex items-center justify-between">
                          <div className="flex items-center gap-2">
                            {result.passed ? (
                              <CheckCircle className="h-5 w-5 text-green-500" />
                            ) : (
                              <XCircle className="h-5 w-5 text-red-500" />
                            )}
                            <span className="font-medium">{result.test}</span>
                          </div>
                          <Badge variant={result.passed ? "default" : "destructive"}>
                            {result.passed ? "Passed" : "Failed"}
                          </Badge>
                        </div>
                        <p className="text-sm text-muted-foreground mt-2">{result.message}</p>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* GitHub Tab */}
          <TabsContent value="github" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>GitHub Integration</CardTitle>
                <CardDescription>Push code and create pull requests</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <Button onClick={() => pushToGitHub("main")} disabled={pushing}>
                    {pushing ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <Github className="h-4 w-4 mr-2" />}
                    Push to GitHub
                  </Button>
                  <Button onClick={() => createPR("feature-branch", "main", "New Feature")} disabled={pushing}>
                    Create Pull Request
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* CI/CD Tab */}
          <TabsContent value="cicd" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>CI/CD Pipeline Generator</CardTitle>
                <CardDescription>Generate GitHub Actions and GitLab CI configurations</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <Button onClick={handleGeneratePipeline} disabled={genPipeline}>
                  {genPipeline ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : <GitBranch className="h-4 w-4 mr-2" />}
                  Generate Pipeline
                </Button>

                <div className="space-y-2">
                  {pipelines.map((pipeline) => (
                    <Card key={pipeline.name}>
                      <CardContent className="p-4">
                        <div className="flex items-center justify-between">
                          <div>
                            <p className="font-medium">{pipeline.name}</p>
                            <p className="text-xs text-muted-foreground">
                              {pipeline.type} • {pipeline.jobs.length} jobs
                            </p>
                          </div>
                          <Badge variant="outline">{pipeline.type}</Badge>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Laravel Runtime Tab */}
          <TabsContent value="laravel" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Laravel Runtime Configuration</CardTitle>
                <CardDescription>Configure PHP version, Composer, and Artisan commands</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                {/* PHP Version */}
                <div className="space-y-2">
                  <Label>PHP Version</Label>
                  <Select
                    value={laravelConfig.phpVersion}
                    onValueChange={(value: "8.1" | "8.2" | "8.3") => 
                      updateLaravelConfig({ phpVersion: value })
                    }
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="8.1">PHP 8.1</SelectItem>
                      <SelectItem value="8.2">PHP 8.2</SelectItem>
                      <SelectItem value="8.3">PHP 8.3</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                {/* Composer Options */}
                <div className="space-y-3">
                  <Label className="text-base font-semibold">Composer Setup</Label>
                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="force-composer"
                      checked={laravelConfig.forceComposerInstall}
                      onCheckedChange={(checked) =>
                        updateLaravelConfig({ forceComposerInstall: checked as boolean })
                      }
                    />
                    <Label htmlFor="force-composer" className="font-normal cursor-pointer">
                      Force composer install (rebuild vendor)
                    </Label>
                  </div>
                </div>

                {/* Artisan Commands */}
                <div className="space-y-3">
                  <Label className="text-base font-semibold">Artisan Commands</Label>
                  <div className="space-y-2">
                    <div className="flex items-center space-x-2">
                      <Checkbox
                        id="run-migrations"
                        checked={laravelConfig.runMigrations}
                        onCheckedChange={(checked) =>
                          updateLaravelConfig({ runMigrations: checked as boolean })
                        }
                      />
                      <Label htmlFor="run-migrations" className="font-normal cursor-pointer">
                        Run migrations (php artisan migrate --force)
                      </Label>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Checkbox
                        id="run-seed"
                        checked={laravelConfig.runSeed}
                        onCheckedChange={(checked) =>
                          updateLaravelConfig({ runSeed: checked as boolean })
                        }
                      />
                      <Label htmlFor="run-seed" className="font-normal cursor-pointer">
                        Run database seeds (php artisan db:seed --force)
                      </Label>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Checkbox
                        id="run-optimize"
                        checked={laravelConfig.runOptimize}
                        onCheckedChange={(checked) =>
                          updateLaravelConfig({ runOptimize: checked as boolean })
                        }
                      />
                      <Label htmlFor="run-optimize" className="font-normal cursor-pointer">
                        Run optimization (caches config, routes, views)
                      </Label>
                    </div>
                  </div>
                </div>

                {/* Action Buttons */}
                <div className="flex gap-3 pt-4 border-t">
                  <Button onClick={handleRunComposer} disabled={installingComposer}>
                    {installingComposer ? (
                      <Loader2 className="h-4 w-4 animate-spin mr-2" />
                    ) : (
                      <Container className="h-4 w-4 mr-2" />
                    )}
                    Run Composer Install
                  </Button>
                  <Button onClick={handleRunArtisan} disabled={runningArtisan} variant="outline">
                    {runningArtisan ? (
                      <Loader2 className="h-4 w-4 animate-spin mr-2" />
                    ) : (
                      <Play className="h-4 w-4 mr-2" />
                    )}
                    Run Artisan Commands
                  </Button>
                </div>
              </CardContent>
            </Card>

            {/* Composer Logs */}
            {composerLogs.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle>Composer Output</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="bg-black text-green-400 p-4 rounded-lg font-mono text-xs h-64 overflow-auto space-y-1">
                    {composerLogs.map((log, idx) => (
                      <p key={idx}>{log}</p>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Artisan Results */}
            {artisanResults.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle>Artisan Command Results</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-2">
                    {artisanResults.map((result, idx) => (
                      <Card key={idx} className={`border-l-4 ${result.success ? 'border-l-green-500' : 'border-l-red-500'}`}>
                        <CardContent className="p-3">
                          <div className="flex items-start justify-between">
                            <div className="flex-1">
                              <div className="flex items-center gap-2">
                                {result.success ? (
                                  <CheckCircle className="h-4 w-4 text-green-500" />
                                ) : (
                                  <XCircle className="h-4 w-4 text-red-500" />
                                )}
                                <code className="text-sm font-mono">{result.command}</code>
                              </div>
                              <p className="text-xs text-muted-foreground mt-1">
                                Duration: {(result.duration / 1000).toFixed(2)}s
                              </p>
                              {result.error && (
                                <p className="text-xs text-red-500 mt-1">{result.error}</p>
                              )}
                            </div>
                          </div>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Artisan Logs */}
            {artisanLogs.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle>Artisan Output</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="bg-black text-green-400 p-4 rounded-lg font-mono text-xs h-64 overflow-auto space-y-1">
                    {artisanLogs.map((log, idx) => (
                      <p key={idx}>{log}</p>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          {/* Console Tab */}
          <TabsContent value="console" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>DevOps Console</CardTitle>
                <CardDescription>Real-time logs and command output</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="bg-black text-green-400 p-4 rounded-lg font-mono text-sm h-96 overflow-auto">
                  <p>[DevOps] System initialized</p>
                  <p>[DevOps] Ready to execute commands</p>
                  <p className="text-yellow-400">[Info] All systems operational</p>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
};

export default StudioDevOps;
