import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Badge } from "@/components/ui/badge";
import { useAutoDocGenerator } from "@/hooks/useAutoDocGenerator";
import { FileText, Search, Database, Globe, Code, RefreshCw } from "lucide-react";
import { formatDistanceToNow } from "date-fns";

export default function StudioDocs() {
  const { generating, docs, generateDocs, getLatestDoc, searchDocs } = useAutoDocGenerator();
  const [projectName, setProjectName] = useState("");
  const [searchQuery, setSearchQuery] = useState("");
  
  const latestDoc = getLatestDoc();
  const searchResults = searchQuery ? searchDocs(searchQuery) : [];

  const handleGenerate = async () => {
    if (!projectName) return;
    await generateDocs(projectName);
    setProjectName("");
  };

  return (
    <div className="container mx-auto p-6 max-w-7xl">
      <div className="flex items-center justify-between mb-8">
        <div>
          <h1 className="text-4xl font-bold mb-2">Auto Documentation</h1>
          <p className="text-muted-foreground">Automatically generated project documentation</p>
        </div>
      </div>

      <Card className="mb-6">
        <CardHeader>
          <CardTitle>Generate Documentation</CardTitle>
          <CardDescription>Create comprehensive docs for any project</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="flex gap-2">
            <Input
              placeholder="Project name (e.g., Smart Friend)"
              value={projectName}
              onChange={(e) => setProjectName(e.target.value)}
              onKeyPress={(e) => e.key === "Enter" && handleGenerate()}
            />
            <Button onClick={handleGenerate} disabled={generating || !projectName}>
              <RefreshCw className={`h-4 w-4 mr-2 ${generating ? 'animate-spin' : ''}`} />
              {generating ? "Generating..." : "Generate"}
            </Button>
          </div>
        </CardContent>
      </Card>

      {docs.length > 0 && (
        <Card className="mb-6">
          <CardHeader>
            <CardTitle>Search Documentation</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="relative">
              <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Search modules, features, routes..."
                className="pl-9"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
              />
            </div>
            
            {searchResults.length > 0 && (
              <div className="mt-4 space-y-2">
                {searchResults.map(doc => (
                  <Card key={doc.id} className="cursor-pointer hover:border-primary">
                    <CardHeader>
                      <CardTitle className="text-base">{doc.projectName}</CardTitle>
                      <CardDescription>
                        Generated {formatDistanceToNow(doc.generatedAt, { addSuffix: true })}
                      </CardDescription>
                    </CardHeader>
                  </Card>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      )}

      {latestDoc ? (
        <Tabs defaultValue="overview" className="space-y-6">
          <TabsList>
            <TabsTrigger value="overview">
              <FileText className="h-4 w-4 mr-2" />
              Overview
            </TabsTrigger>
            <TabsTrigger value="modules">
              <Code className="h-4 w-4 mr-2" />
              Modules ({latestDoc.modules.length})
            </TabsTrigger>
            <TabsTrigger value="routes">
              <Globe className="h-4 w-4 mr-2" />
              Routes ({latestDoc.routes.length})
            </TabsTrigger>
            <TabsTrigger value="database">
              <Database className="h-4 w-4 mr-2" />
              Database ({latestDoc.database.totalTables})
            </TabsTrigger>
          </TabsList>

          <TabsContent value="overview">
            <Card>
              <CardHeader>
                <CardTitle>{latestDoc.projectName}</CardTitle>
                <CardDescription>
                  Version {latestDoc.version} • Generated {formatDistanceToNow(latestDoc.generatedAt, { addSuffix: true })}
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div>
                  <h3 className="font-semibold mb-2">Overview</h3>
                  <p className="text-muted-foreground">{latestDoc.overview}</p>
                </div>

                <div className="grid grid-cols-3 gap-4">
                  <Card>
                    <CardHeader className="pb-3">
                      <CardTitle className="text-sm">Modules</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="text-3xl font-bold">{latestDoc.modules.length}</div>
                    </CardContent>
                  </Card>
                  
                  <Card>
                    <CardHeader className="pb-3">
                      <CardTitle className="text-sm">Routes</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="text-3xl font-bold">{latestDoc.routes.length}</div>
                    </CardContent>
                  </Card>
                  
                  <Card>
                    <CardHeader className="pb-3">
                      <CardTitle className="text-sm">DB Tables</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="text-3xl font-bold">{latestDoc.database.totalTables}</div>
                    </CardContent>
                  </Card>
                </div>

                <div>
                  <h3 className="font-semibold mb-3">API Endpoints</h3>
                  <div className="space-y-2">
                    {latestDoc.apiEndpoints.map((endpoint, index) => (
                      <Card key={index}>
                        <CardContent className="pt-6">
                          <div className="flex items-start justify-between">
                            <div className="flex-1">
                              <div className="flex items-center gap-2 mb-1">
                                <Badge variant="outline">{endpoint.method}</Badge>
                                <code className="text-sm">{endpoint.path}</code>
                              </div>
                              <p className="text-sm text-muted-foreground">{endpoint.description}</p>
                            </div>
                          </div>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="modules">
            <div className="space-y-4">
              {latestDoc.modules.map((module, index) => (
                <Card key={index}>
                  <CardHeader>
                    <CardTitle>{module.name}</CardTitle>
                    <CardDescription>{module.description}</CardDescription>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <div>
                      <h4 className="font-semibold mb-2">Features</h4>
                      <div className="flex flex-wrap gap-2">
                        {module.features.map((feature, i) => (
                          <Badge key={i} variant="secondary">{feature}</Badge>
                        ))}
                      </div>
                    </div>

                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <h4 className="font-semibold mb-2">Pages</h4>
                        <ScrollArea className="h-32">
                          <div className="space-y-1">
                            {module.pages.map((page, i) => (
                              <code key={i} className="block text-sm bg-muted p-2 rounded">
                                {page}
                              </code>
                            ))}
                          </div>
                        </ScrollArea>
                      </div>

                      <div>
                        <h4 className="font-semibold mb-2">Components</h4>
                        <ScrollArea className="h-32">
                          <div className="space-y-1">
                            {module.components.map((comp, i) => (
                              <code key={i} className="block text-sm bg-muted p-2 rounded">
                                {comp}
                              </code>
                            ))}
                          </div>
                        </ScrollArea>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>

          <TabsContent value="routes">
            <div className="space-y-2">
              {latestDoc.routes.map((route, index) => (
                <Card key={index}>
                  <CardContent className="pt-6">
                    <div className="flex items-center justify-between">
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-1">
                          <code className="text-sm font-mono">{route.path}</code>
                          {route.protected && <Badge variant="destructive" className="text-xs">Protected</Badge>}
                        </div>
                        <p className="text-sm text-muted-foreground">{route.description}</p>
                        {route.component && (
                          <p className="text-xs text-muted-foreground mt-1">
                            Component: {route.component}
                          </p>
                        )}
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>

          <TabsContent value="database">
            <div className="space-y-4">
              {latestDoc.database.tables.map((table, index) => (
                <Card key={index}>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <Database className="h-5 w-5" />
                      {table.name}
                    </CardTitle>
                    <CardDescription>{table.description}</CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-2">
                      {table.columns.map((column, i) => (
                        <div key={i} className="flex items-center justify-between p-2 bg-muted rounded">
                          <div className="flex items-center gap-2">
                            <code className="text-sm font-mono">{column.name}</code>
                            <Badge variant="outline" className="text-xs">{column.type}</Badge>
                            {!column.nullable && <Badge variant="secondary" className="text-xs">Required</Badge>}
                          </div>
                          {column.description && (
                            <p className="text-xs text-muted-foreground">{column.description}</p>
                          )}
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>
        </Tabs>
      ) : (
        <Card>
          <CardContent className="flex flex-col items-center justify-center py-12">
            <FileText className="h-12 w-12 text-muted-foreground mb-4" />
            <p className="text-muted-foreground">No documentation generated yet</p>
            <p className="text-sm text-muted-foreground">Enter a project name above to get started</p>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
