import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { useAutoBugFinder } from "@/hooks/useAutoBugFinder";
import { AlertTriangle, Bug, Wrench, FileCode, RefreshCw } from "lucide-react";
import { formatDistanceToNow } from "date-fns";

export default function StudioErrors() {
  const { 
    scanning, 
    fixing, 
    bugs, 
    scanForErrors, 
    applyAutoFix, 
    getCriticalBugs 
  } = useAutoBugFinder();

  const [selectedBug, setSelectedBug] = useState<string | null>(null);
  const criticalBugs = getCriticalBugs();

  useEffect(() => {
    // Auto-scan on mount
    scanForErrors();
  }, []);

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case "critical": return "destructive";
      case "error": return "default";
      case "warning": return "secondary";
      case "info": return "outline";
      default: return "secondary";
    }
  };

  const getSeverityIcon = (severity: string) => {
    switch (severity) {
      case "critical": return <AlertTriangle className="h-5 w-5 text-red-500" />;
      case "error": return <Bug className="h-5 w-5 text-orange-500" />;
      case "warning": return <AlertTriangle className="h-5 w-5 text-amber-500" />;
      default: return <Bug className="h-5 w-5 text-blue-500" />;
    }
  };

  return (
    <div className="container mx-auto p-6 max-w-7xl">
      <div className="flex items-center justify-between mb-8">
        <div>
          <h1 className="text-4xl font-bold mb-2">Error Dashboard</h1>
          <p className="text-muted-foreground">Auto bug finder and fixer</p>
        </div>
        <Button onClick={() => scanForErrors()} disabled={scanning}>
          <RefreshCw className={`h-4 w-4 mr-2 ${scanning ? 'animate-spin' : ''}`} />
          {scanning ? "Scanning..." : "Scan Project"}
        </Button>
      </div>

      {criticalBugs.length > 0 && (
        <Card className="mb-6 border-red-500 bg-red-50 dark:bg-red-950">
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-red-700 dark:text-red-300">
              <AlertTriangle className="h-5 w-5" />
              Critical Issues Detected
            </CardTitle>
            <CardDescription>
              {criticalBugs.length} critical issue{criticalBugs.length > 1 ? 's' : ''} requiring immediate attention
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              {criticalBugs.map(bug => (
                <div key={bug.id} className="flex items-center justify-between bg-background p-3 rounded">
                  <div className="flex-1">
                    <p className="font-medium">{bug.errorMessage.substring(0, 80)}...</p>
                    <p className="text-sm text-muted-foreground">{bug.affectedFiles[0]}</p>
                  </div>
                  <Button 
                    onClick={() => applyAutoFix(bug.id)} 
                    disabled={fixing === bug.id}
                    size="sm"
                  >
                    <Wrench className="h-4 w-4 mr-2" />
                    {fixing === bug.id ? "Fixing..." : "Auto Fix"}
                  </Button>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      <div className="grid grid-cols-3 gap-6">
        <Card>
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium">Total Issues</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-3xl font-bold">{bugs.length}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium">Critical</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-3xl font-bold text-red-500">{criticalBugs.length}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium">Files Affected</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-3xl font-bold">
              {new Set(bugs.flatMap(b => b.affectedFiles)).size}
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="mt-6 space-y-4">
        {bugs.length === 0 && !scanning ? (
          <Card>
            <CardContent className="flex flex-col items-center justify-center py-12">
              <Bug className="h-12 w-12 text-muted-foreground mb-4" />
              <p className="text-muted-foreground">No bugs found! Your code is clean.</p>
            </CardContent>
          </Card>
        ) : (
          bugs.map(bug => (
            <Card key={bug.id} className="cursor-pointer hover:border-primary transition-colors" onClick={() => setSelectedBug(selectedBug === bug.id ? null : bug.id)}>
              <CardHeader>
                <div className="flex items-start justify-between">
                  <div className="flex items-start gap-3 flex-1">
                    {getSeverityIcon(bug.severity)}
                    <div className="flex-1">
                      <CardTitle className="text-base mb-2">{bug.errorMessage}</CardTitle>
                      <div className="flex flex-wrap gap-2 text-sm text-muted-foreground">
                        <div className="flex items-center gap-1">
                          <FileCode className="h-4 w-4" />
                          <span>{bug.affectedFiles.length} file{bug.affectedFiles.length > 1 ? 's' : ''}</span>
                        </div>
                        <span>•</span>
                        <span>{bug.occurrences} occurrence{bug.occurrences > 1 ? 's' : ''}</span>
                        <span>•</span>
                        <span>Last seen {formatDistanceToNow(bug.lastSeen, { addSuffix: true })}</span>
                      </div>
                    </div>
                  </div>
                  <div className="flex gap-2">
                    <Badge variant={getSeverityColor(bug.severity)}>{bug.severity}</Badge>
                  </div>
                </div>
              </CardHeader>

              {selectedBug === bug.id && (
                <CardContent className="space-y-4">
                  <div>
                    <h4 className="font-semibold mb-2">Affected Files:</h4>
                    <div className="space-y-1">
                      {bug.affectedFiles.map((file, index) => (
                        <div key={index} className="text-sm font-mono bg-muted p-2 rounded">
                          {file}
                        </div>
                      ))}
                    </div>
                  </div>

                  {bug.stackTrace && (
                    <div>
                      <h4 className="font-semibold mb-2">Stack Trace:</h4>
                      <div className="text-sm font-mono bg-muted p-3 rounded">
                        {bug.stackTrace}
                      </div>
                    </div>
                  )}

                  <div>
                    <h4 className="font-semibold mb-2">Suggested Fixes:</h4>
                    <ScrollArea className="h-40">
                      <div className="space-y-2">
                        {bug.suggestedFixes.map((fix, index) => (
                          <div key={index} className="text-sm bg-muted p-3 rounded flex items-start gap-2">
                            <span className="text-primary font-bold">{index + 1}.</span>
                            <span>{fix}</span>
                          </div>
                        ))}
                      </div>
                    </ScrollArea>
                  </div>

                  <Button 
                    onClick={(e) => {
                      e.stopPropagation();
                      applyAutoFix(bug.id);
                    }} 
                    disabled={fixing === bug.id}
                    className="w-full"
                  >
                    <Wrench className="h-4 w-4 mr-2" />
                    {fixing === bug.id ? "Applying Fix..." : "Auto Fix with Smart Friend"}
                  </Button>
                </CardContent>
              )}
            </Card>
          ))
        )}
      </div>
    </div>
  );
}
