import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { useModuleGenerator } from "@/hooks/useModuleGenerator";
import { WalletGuard } from "@/components/ide/WalletGuard";
import { Package, Loader2, Search, Grid, List } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { cn } from "@/lib/utils";

const StudioModules = () => {
  const [projectId] = useState("default-project");
  const [selectedTemplate, setSelectedTemplate] = useState<string | null>(null);
  const [customName, setCustomName] = useState("");
  const [searchQuery, setSearchQuery] = useState("");
  const [viewMode, setViewMode] = useState<"grid" | "list">("grid");
  const { templates, isLoading, generateModule } = useModuleGenerator();

  const filteredTemplates = templates.filter(t => 
    t.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
    t.category.toLowerCase().includes(searchQuery.toLowerCase()) ||
    t.description?.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const categories = Array.from(new Set(templates.map(t => t.category)));

  const handleGenerate = async () => {
    if (!selectedTemplate) return;
    await generateModule.mutateAsync({ 
      templateSlug: selectedTemplate, 
      projectId,
      customName: customName || undefined 
    });
    setSelectedTemplate(null);
    setCustomName("");
  };

  if (isLoading) {
    return (
      <div className="h-screen flex items-center justify-center">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  return (
    <WalletGuard>
      <div className="h-screen bg-gradient-to-br from-primary/5 via-background to-secondary/5 p-6 overflow-y-auto">
        <div className="max-w-7xl mx-auto space-y-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <div className="p-3 bg-gradient-to-br from-primary/20 to-secondary/20 rounded-xl">
                <Package className="h-8 w-8 text-primary" />
              </div>
              <div>
                <h1 className="text-3xl font-bold bg-gradient-to-r from-primary to-secondary bg-clip-text text-transparent">
                  Module Builder
                </h1>
                <p className="text-muted-foreground">50+ Ready-to-Generate Modules</p>
              </div>
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant={viewMode === "grid" ? "default" : "outline"}
                size="icon"
                onClick={() => setViewMode("grid")}
              >
                <Grid className="h-4 w-4" />
              </Button>
              <Button
                variant={viewMode === "list" ? "default" : "outline"}
                size="icon"
                onClick={() => setViewMode("list")}
              >
                <List className="h-4 w-4" />
              </Button>
            </div>
          </div>

          <div className="relative">
            <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search modules..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10"
            />
          </div>

          <Tabs defaultValue="all" className="w-full">
            <TabsList className="w-full justify-start overflow-x-auto">
              <TabsTrigger value="all">All ({templates.length})</TabsTrigger>
              {categories.map(cat => (
                <TabsTrigger key={cat} value={cat}>
                  {cat} ({templates.filter(t => t.category === cat).length})
                </TabsTrigger>
              ))}
            </TabsList>

            <TabsContent value="all" className="mt-6">
              <div className={cn(
                viewMode === "grid" 
                  ? "grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4" 
                  : "space-y-4"
              )}>
                {filteredTemplates.map((template) => (
                  <Card 
                    key={template.id}
                    className={cn(
                      "cursor-pointer transition-all hover:shadow-xl",
                      selectedTemplate === template.slug && "ring-2 ring-primary"
                    )}
                    onClick={() => setSelectedTemplate(template.slug)}
                  >
                    <CardHeader>
                      <div className="flex items-start justify-between">
                        <CardTitle className="text-lg">{template.name}</CardTitle>
                        <Badge variant="secondary">{template.category}</Badge>
                      </div>
                      <CardDescription className="line-clamp-2">
                        {template.description}
                      </CardDescription>
                    </CardHeader>
                    {selectedTemplate === template.slug && (
                      <CardContent className="space-y-3">
                        <Input
                          placeholder="Custom name (optional)"
                          value={customName}
                          onChange={(e) => setCustomName(e.target.value)}
                          onClick={(e) => e.stopPropagation()}
                        />
                        <div className="flex gap-2">
                          <Button 
                            onClick={(e) => {
                              e.stopPropagation();
                              handleGenerate();
                            }}
                            className="flex-1"
                            disabled={generateModule.isPending}
                          >
                            {generateModule.isPending ? (
                              <>
                                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                                Generating...
                              </>
                            ) : (
                              "Generate Module"
                            )}
                          </Button>
                          <Button
                            variant="outline"
                            onClick={(e) => {
                              e.stopPropagation();
                              setSelectedTemplate(null);
                              setCustomName("");
                            }}
                          >
                            Cancel
                          </Button>
                        </div>
                      </CardContent>
                    )}
                  </Card>
                ))}
              </div>
            </TabsContent>

            {categories.map(cat => (
              <TabsContent key={cat} value={cat} className="mt-6">
                <div className={cn(
                  viewMode === "grid" 
                    ? "grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4" 
                    : "space-y-4"
                )}>
                  {filteredTemplates
                    .filter(t => t.category === cat)
                    .map((template) => (
                      <Card 
                        key={template.id}
                        className={cn(
                          "cursor-pointer transition-all hover:shadow-xl",
                          selectedTemplate === template.slug && "ring-2 ring-primary"
                        )}
                        onClick={() => setSelectedTemplate(template.slug)}
                      >
                        <CardHeader>
                          <CardTitle className="text-lg">{template.name}</CardTitle>
                          <CardDescription className="line-clamp-2">
                            {template.description}
                          </CardDescription>
                        </CardHeader>
                        {selectedTemplate === template.slug && (
                          <CardContent className="space-y-3">
                            <Input
                              placeholder="Custom name (optional)"
                              value={customName}
                              onChange={(e) => setCustomName(e.target.value)}
                              onClick={(e) => e.stopPropagation()}
                            />
                            <div className="flex gap-2">
                              <Button 
                                onClick={(e) => {
                                  e.stopPropagation();
                                  handleGenerate();
                                }}
                                className="flex-1"
                                disabled={generateModule.isPending}
                              >
                                {generateModule.isPending ? (
                                  <>
                                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                                    Generating...
                                  </>
                                ) : (
                                  "Generate Module"
                                )}
                              </Button>
                              <Button
                                variant="outline"
                                onClick={(e) => {
                                  e.stopPropagation();
                                  setSelectedTemplate(null);
                                  setCustomName("");
                                }}
                              >
                                Cancel
                              </Button>
                            </div>
                          </CardContent>
                        )}
                      </Card>
                    ))}
                </div>
              </TabsContent>
            ))}
          </Tabs>
        </div>
      </div>
    </WalletGuard>
  );
};

export default StudioModules;
