import { useState } from "react";
import { WalletGuard } from "@/components/ide/WalletGuard";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { useOfflineBrain } from "@/hooks/useOfflineBrain";
import { useIdentityEngine } from "@/hooks/useIdentityEngine";
import { useSafetyFilter } from "@/hooks/useSafetyFilter";
import { Database, Shield, Brain, Wifi, WifiOff } from "lucide-react";

const StudioOfflineSettings = () => {
  const { toast } = useToast();
  const { offlineBrain, saveOfflineResponse } = useOfflineBrain();
  const { role, IDENTITY_RULES } = useIdentityEngine();
  const { getSafetyGuidelines } = useSafetyFilter();
  
  const [offlineMode, setOfflineMode] = useState(false);
  const [newQuestion, setNewQuestion] = useState("");
  const [newAnswer, setNewAnswer] = useState("");
  const [newCategory, setNewCategory] = useState("general");

  const handleSaveOfflineResponse = async () => {
    if (!newQuestion || !newAnswer) {
      toast({
        title: "Error",
        description: "Please fill in both question and answer",
        variant: "destructive",
      });
      return;
    }

    await saveOfflineResponse(newQuestion, newAnswer, newCategory);
    
    toast({
      title: "Success",
      description: "Offline response saved successfully",
    });

    setNewQuestion("");
    setNewAnswer("");
    setNewCategory("general");
  };

  return (
    <WalletGuard>
      <div className="h-screen bg-gradient-to-br from-primary/5 via-background to-secondary/5 p-6 overflow-y-auto">
        <div className="max-w-6xl mx-auto space-y-6">
          {/* Header */}
          <div className="flex items-center gap-3">
            <div className="p-3 bg-gradient-to-br from-primary/20 to-secondary/20 rounded-xl">
              <Brain className="h-8 w-8 text-primary" />
            </div>
            <div>
              <h1 className="text-3xl font-bold bg-gradient-to-r from-primary to-secondary bg-clip-text text-transparent">
                Offline Brain Settings
              </h1>
              <p className="text-muted-foreground">Manage offline responses, identity rules, and safety filters</p>
            </div>
          </div>

          {/* Offline Mode Toggle */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                {offlineMode ? <WifiOff className="h-5 w-5" /> : <Wifi className="h-5 w-5" />}
                Offline Mode
              </CardTitle>
              <CardDescription>
                Enable offline mode to use cached responses when internet is unavailable
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="flex items-center space-x-2">
                <Switch
                  id="offline-mode"
                  checked={offlineMode}
                  onCheckedChange={setOfflineMode}
                />
                <Label htmlFor="offline-mode">
                  {offlineMode ? "Offline Mode Active" : "Online Mode Active"}
                </Label>
              </div>
            </CardContent>
          </Card>

          {/* Role & Safety Status */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Shield className="h-5 w-5" />
                Current Role & Safety Level
              </CardTitle>
              <CardDescription>
                Switch between Owner and Ava modes in the Chat page
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <div className="p-3 bg-muted rounded-lg">
                  <div className="font-medium text-sm mb-1">Active Role</div>
                  <div className="text-sm text-muted-foreground">
                    {role === 'owner' ? '👨‍💼 Owner Mode (Professional)' : '👶 Ava Mode (Child-Safe)'}
                  </div>
                </div>
                <div className="p-3 bg-muted rounded-lg">
                  <div className="font-medium text-sm mb-1">Safety Guidelines</div>
                  <div className="text-xs text-muted-foreground whitespace-pre-line">
                    {getSafetyGuidelines(role, 'en')}
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Add Offline Response */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Database className="h-5 w-5" />
                Add Offline Response
              </CardTitle>
              <CardDescription>
                Create custom offline responses for common questions
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <Label>Question</Label>
                <Input
                  placeholder="Enter question"
                  value={newQuestion}
                  onChange={(e) => setNewQuestion(e.target.value)}
                />
              </div>
              <div>
                <Label>Answer</Label>
                <Textarea
                  placeholder="Enter answer"
                  value={newAnswer}
                  onChange={(e) => setNewAnswer(e.target.value)}
                  rows={4}
                />
              </div>
              <div>
                <Label>Category</Label>
                <Input
                  placeholder="general, greeting, identity, etc."
                  value={newCategory}
                  onChange={(e) => setNewCategory(e.target.value)}
                />
              </div>
              <Button onClick={handleSaveOfflineResponse}>
                Save Offline Response
              </Button>
            </CardContent>
          </Card>

          {/* Offline Brain Responses */}
          <Card>
            <CardHeader>
              <CardTitle>Cached Offline Responses ({offlineBrain.length})</CardTitle>
              <CardDescription>
                These responses will be used when offline mode is active
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3 max-h-96 overflow-y-auto">
                {offlineBrain.map((entry) => (
                  <div key={entry.id} className="p-3 bg-muted rounded-lg">
                    <div className="font-medium text-sm">{entry.question}</div>
                    <div className="text-sm text-muted-foreground mt-1">{entry.answer}</div>
                    <div className="text-xs text-muted-foreground mt-2">
                      Category: {entry.category} • Language: {entry.language}
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          {/* Identity & Privacy Protection */}
          <Card>
            <CardHeader>
              <CardTitle>Identity & Privacy Protection</CardTitle>
              <CardDescription>
                Hard-coded identity rules that protect owner information (Phase I)
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <div className="p-3 bg-primary/5 border border-primary/20 rounded-lg">
                  <div className="font-medium text-sm mb-2 flex items-center gap-2">
                    <Shield className="h-4 w-4 text-primary" />
                    Owner Identity (Bengali)
                  </div>
                  <div className="text-sm text-muted-foreground">
                    {IDENTITY_RULES.ownerIdentityLine.bn}
                  </div>
                </div>
                
                <div className="p-3 bg-primary/5 border border-primary/20 rounded-lg">
                  <div className="font-medium text-sm mb-2 flex items-center gap-2">
                    <Shield className="h-4 w-4 text-primary" />
                    Owner Identity (English)
                  </div>
                  <div className="text-sm text-muted-foreground">
                    {IDENTITY_RULES.ownerIdentityLine.en}
                  </div>
                </div>

                <div className="p-3 bg-destructive/5 border border-destructive/20 rounded-lg">
                  <div className="font-medium text-sm mb-2 flex items-center gap-2">
                    <Shield className="h-4 w-4 text-destructive" />
                    Privacy Refusal (Bengali)
                  </div>
                  <div className="text-sm text-muted-foreground">
                    {IDENTITY_RULES.privacyRefusal.bn}
                  </div>
                </div>

                <div className="p-3 bg-destructive/5 border border-destructive/20 rounded-lg">
                  <div className="font-medium text-sm mb-2 flex items-center gap-2">
                    <Shield className="h-4 w-4 text-destructive" />
                    Privacy Refusal (English)
                  </div>
                  <div className="text-sm text-muted-foreground">
                    {IDENTITY_RULES.privacyRefusal.en}
                  </div>
                </div>

                <div className="p-3 bg-muted rounded-lg">
                  <div className="font-medium text-sm mb-2">Protected Information</div>
                  <ul className="text-xs text-muted-foreground space-y-1">
                    <li>• Personal addresses and locations</li>
                    <li>• Phone numbers and contact details</li>
                    <li>• NID/passport information</li>
                    <li>• Financial account details</li>
                  </ul>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Safety Pattern Information */}
          <Card>
            <CardHeader>
              <CardTitle>Active Safety Patterns</CardTitle>
              <CardDescription>
                Pattern-based content filtering for child safety (Phase I)
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <div className="p-3 bg-destructive/5 border border-destructive/20 rounded-lg">
                  <div className="font-medium text-sm mb-2 text-destructive">Level 3 (Ava Mode - Strict)</div>
                  <ul className="text-xs text-muted-foreground space-y-1">
                    <li>• Adult content blocked</li>
                    <li>• Violence and weapons blocked</li>
                    <li>• Drugs and substances blocked</li>
                    <li>• Self-harm content blocked</li>
                    <li>• Horror and scary content blocked</li>
                  </ul>
                </div>

                <div className="p-3 bg-orange-500/5 border border-orange-500/20 rounded-lg">
                  <div className="font-medium text-sm mb-2 text-orange-600">Level 2 (Medium)</div>
                  <ul className="text-xs text-muted-foreground space-y-1">
                    <li>• Political extremism blocked</li>
                    <li>• Hate speech blocked</li>
                  </ul>
                </div>

                <div className="p-3 bg-yellow-500/5 border border-yellow-500/20 rounded-lg">
                  <div className="font-medium text-sm mb-2 text-yellow-600">Level 1 (Owner Mode - Normal)</div>
                  <ul className="text-xs text-muted-foreground space-y-1">
                    <li>• Criminal activities blocked</li>
                    <li>• Illegal content guidance blocked</li>
                  </ul>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </WalletGuard>
  );
};

export default StudioOfflineSettings;
