import { useState } from "react";
import { Search, Sparkles, Globe, Loader2, ExternalLink, Clock } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import type { SearchResponse, SearchResult } from "@/types/search";

const StudioSearch = () => {
  const [query, setQuery] = useState("");
  const [lang, setLang] = useState("bn");
  const [module, setModule] = useState("all");
  const [isSearching, setIsSearching] = useState(false);
  const [searchResults, setSearchResults] = useState<SearchResponse | null>(null);
  const [useAI, setUseAI] = useState(false);
  const [aiResponse, setAiResponse] = useState("");
  const [isLoadingAI, setIsLoadingAI] = useState(false);
  const { toast } = useToast();

  const modules = [
    { value: "all", label: "All Modules" },
    { value: "academy", label: "Smart Academy" },
    { value: "news", label: "Smart News" },
    { value: "doctor", label: "Smart Doctor" },
    { value: "lawyer", label: "Smart Lawyer" },
    { value: "tourist", label: "Smart Tourist" },
    { value: "custom", label: "Custom" },
  ];

  const handleSearch = async () => {
    if (!query.trim()) {
      toast({ title: "Please enter a search query", variant: "destructive" });
      return;
    }

    setIsSearching(true);
    setSearchResults(null);
    setAiResponse("");

    try {
      const { data, error } = await supabase.functions.invoke("web-search", {
        body: {
          query: query.trim(),
          lang,
          maxResults: 10,
          module: module === "all" ? undefined : module,
          safeSearch: true,
        },
      });

      if (error) throw error;

      setSearchResults(data as SearchResponse);

      // If AI is enabled, get AI summary
      if (useAI && data.results.length > 0) {
        setIsLoadingAI(true);
        try {
          const context = data.results
            .slice(0, 5)
            .map((r: SearchResult) => `${r.title}\n${r.snippet}\nSource: ${r.url}`)
            .join("\n\n");

          const aiPrompt = `Based on these web search results, provide a comprehensive answer to: "${query}"\n\nSearch Results:\n${context}`;

          const { data: aiData, error: aiError } = await supabase.functions.invoke("smart-friend-chat", {
            body: {
              messages: [{ role: "user", content: aiPrompt }],
              language: lang,
            },
          });

          if (aiError) throw aiError;

          // Handle streaming response
          if (aiData) {
            let fullResponse = "";
            const reader = aiData.getReader();
            const decoder = new TextDecoder();

            while (true) {
              const { done, value } = await reader.read();
              if (done) break;

              const chunk = decoder.decode(value);
              const lines = chunk.split("\n");

              for (const line of lines) {
                if (line.startsWith("data: ")) {
                  const data = line.slice(6);
                  if (data === "[DONE]") continue;
                  try {
                    const parsed = JSON.parse(data);
                    const content = parsed.choices?.[0]?.delta?.content;
                    if (content) {
                      fullResponse += content;
                      setAiResponse(fullResponse);
                    }
                  } catch (e) {
                    // Ignore parse errors
                  }
                }
              }
            }
          }
        } catch (aiError) {
          console.error("AI error:", aiError);
          toast({ title: "AI summary failed", description: "Showing web results only", variant: "destructive" });
        } finally {
          setIsLoadingAI(false);
        }
      }

      toast({ title: "Search completed", description: `Found ${data.results.length} results in ${data.tookMs}ms` });
    } catch (error: any) {
      console.error("Search error:", error);
      toast({ title: "Search failed", description: error.message, variant: "destructive" });
    } finally {
      setIsSearching(false);
    }
  };

  return (
    <div className="h-full overflow-auto bg-gradient-to-br from-primary/5 via-background to-secondary/5">
      <div className="container mx-auto p-6 max-w-7xl">
        <div className="mb-8">
          <div className="flex items-center gap-3 mb-2">
            <div className="p-2 bg-primary/20 rounded-xl">
              <Globe className="h-6 w-6 text-primary" />
            </div>
            <h1 className="text-3xl font-bold bg-gradient-to-r from-primary to-secondary bg-clip-text text-transparent">
              Smart Search Engine
            </h1>
          </div>
          <p className="text-muted-foreground">AI-powered web search for Academy, News, Doctor and more</p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Main Search Panel */}
          <div className="lg:col-span-2 space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Search the Web</CardTitle>
                <CardDescription>Enter your query to search across the internet</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex gap-2">
                  <Input
                    placeholder="Enter your search query..."
                    value={query}
                    onChange={(e) => setQuery(e.target.value)}
                    onKeyDown={(e) => e.key === "Enter" && handleSearch()}
                    className="flex-1"
                  />
                  <Button onClick={handleSearch} disabled={isSearching}>
                    {isSearching ? <Loader2 className="h-4 w-4 animate-spin" /> : <Search className="h-4 w-4" />}
                    Search
                  </Button>
                </div>

                <div className="flex gap-4">
                  <div className="flex-1">
                    <Label>Language</Label>
                    <Select value={lang} onValueChange={setLang}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="bn">বাংলা (Bangla)</SelectItem>
                        <SelectItem value="en">English</SelectItem>
                        <SelectItem value="hi">हिन्दी (Hindi)</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="flex-1">
                    <Label>Module</Label>
                    <Select value={module} onValueChange={setModule}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        {modules.map((m) => (
                          <SelectItem key={m.value} value={m.value}>
                            {m.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* AI Summary */}
            {useAI && (aiResponse || isLoadingAI) && (
              <Card className="border-primary/20 bg-primary/5">
                <CardHeader>
                  <div className="flex items-center gap-2">
                    <Sparkles className="h-5 w-5 text-primary" />
                    <CardTitle>AI Summary</CardTitle>
                  </div>
                </CardHeader>
                <CardContent>
                  {isLoadingAI ? (
                    <div className="flex items-center gap-2 text-muted-foreground">
                      <Loader2 className="h-4 w-4 animate-spin" />
                      Generating AI summary...
                    </div>
                  ) : (
                    <div className="prose prose-sm dark:prose-invert max-w-none">
                      <p className="whitespace-pre-wrap">{aiResponse}</p>
                    </div>
                  )}
                </CardContent>
              </Card>
            )}

            {/* Search Results */}
            {searchResults && (
              <Card>
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <div>
                      <CardTitle>Search Results</CardTitle>
                      <CardDescription>
                        Found {searchResults.results.length} results
                        {searchResults.module && ` in ${searchResults.module}`}
                      </CardDescription>
                    </div>
                    <Badge variant="outline" className="gap-1">
                      <Clock className="h-3 w-3" />
                      {searchResults.tookMs}ms
                    </Badge>
                  </div>
                </CardHeader>
                <CardContent className="space-y-4">
                  {searchResults.results.map((result, idx) => (
                    <div key={idx} className="p-4 border rounded-lg hover:bg-accent/50 transition-colors">
                      <div className="flex items-start justify-between gap-2">
                        <div className="flex-1 space-y-1">
                          <a
                            href={result.url}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="text-lg font-semibold text-primary hover:underline flex items-center gap-2"
                          >
                            {result.title}
                            <ExternalLink className="h-4 w-4" />
                          </a>
                          <p className="text-sm text-muted-foreground">{result.snippet}</p>
                          <div className="flex items-center gap-2 text-xs">
                            <Badge variant="secondary">{result.source}</Badge>
                            {result.publishedDate && (
                              <span className="text-muted-foreground">
                                {new Date(result.publishedDate).toLocaleDateString()}
                              </span>
                            )}
                            {result.score && (
                              <span className="text-muted-foreground">Score: {(result.score * 100).toFixed(0)}%</span>
                            )}
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </CardContent>
              </Card>
            )}
          </div>

          {/* Side Panel */}
          <div className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">AI Integration</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <Label htmlFor="use-ai" className="cursor-pointer">
                    Also ask Smart Friend AI
                  </Label>
                  <Switch id="use-ai" checked={useAI} onCheckedChange={setUseAI} />
                </div>
                <p className="text-xs text-muted-foreground">
                  When enabled, Smart Friend will analyze search results and provide an AI-generated summary
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="text-lg">Search Tips</CardTitle>
              </CardHeader>
              <CardContent className="space-y-2 text-sm text-muted-foreground">
                <p>• Use specific keywords for better results</p>
                <p>• Select a module to filter results</p>
                <p>• Enable AI summary for instant insights</p>
                <p>• Results are cached for faster access</p>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
};

export default StudioSearch;
