import { useState, useEffect } from "react";
import { Settings, Shield, Zap, Globe } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "@/hooks/useAuth";

const StudioSearchSettings = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [isOwner, setIsOwner] = useState(false);
  const [loading, setLoading] = useState(true);

  const [settings, setSettings] = useState({
    enabled: true,
    maxResults: 5,
    safeSearch: true,
    rateLimit: 10,
    modules: {
      academy: true,
      news: true,
      doctor: true,
      lawyer: true,
      tourist: true,
      salon: false,
      hospital: false,
      bazar: false,
      transport: false,
      restaurant: false,
    },
  });

  useEffect(() => {
    checkOwnerStatus();
    loadSettings();
  }, [user]);

  const checkOwnerStatus = async () => {
    if (!user) return;

    try {
      const { data, error } = await supabase.rpc("is_core_brain_owner");
      if (error) throw error;
      setIsOwner(data as boolean);
    } catch (error) {
      console.error("Error checking owner status:", error);
    }
  };

  const loadSettings = async () => {
    try {
      const { data, error } = await supabase
        .from("user_memory")
        .select("memory_value")
        .eq("memory_key", "search_settings")
        .eq("memory_type", "system")
        .maybeSingle();

      if (error) throw error;

      if (data?.memory_value) {
        setSettings(data.memory_value as any);
      }
    } catch (error) {
      console.error("Error loading settings:", error);
    } finally {
      setLoading(false);
    }
  };

  const saveSettings = async () => {
    if (!user) {
      toast({ title: "Please sign in to save settings", variant: "destructive" });
      return;
    }

    if (!isOwner) {
      toast({ title: "Only owners can change search settings", variant: "destructive" });
      return;
    }

    try {
      const { error } = await supabase.from("user_memory").upsert({
        user_id: user.id,
        memory_key: "search_settings",
        memory_type: "system",
        memory_value: settings,
      });

      if (error) throw error;

      toast({ title: "Settings saved successfully" });
    } catch (error: any) {
      console.error("Error saving settings:", error);
      toast({ title: "Failed to save settings", description: error.message, variant: "destructive" });
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-full">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary mx-auto mb-4"></div>
          <p className="text-muted-foreground">Loading settings...</p>
        </div>
      </div>
    );
  }

  if (!isOwner) {
    return (
      <div className="flex items-center justify-center h-full">
        <Card className="max-w-md">
          <CardHeader>
            <CardTitle>Access Restricted</CardTitle>
            <CardDescription>Only system owners can access search settings</CardDescription>
          </CardHeader>
          <CardContent>
            <p className="text-sm text-muted-foreground">
              Web search settings can only be modified by authorized administrators.
            </p>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="h-full overflow-auto bg-gradient-to-br from-primary/5 via-background to-secondary/5">
      <div className="container mx-auto p-6 max-w-4xl">
        <div className="mb-8">
          <div className="flex items-center gap-3 mb-2">
            <div className="p-2 bg-primary/20 rounded-xl">
              <Settings className="h-6 w-6 text-primary" />
            </div>
            <h1 className="text-3xl font-bold">Web Search Settings</h1>
          </div>
          <p className="text-muted-foreground">Configure web search & safety controls</p>
        </div>

        <div className="space-y-6">
          {/* Master Controls */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-2">
                <Globe className="h-5 w-5" />
                <CardTitle>Master Controls</CardTitle>
              </div>
              <CardDescription>Global web search settings</CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center justify-between">
                <div>
                  <Label htmlFor="enabled" className="text-base">Enable Web Search Globally</Label>
                  <p className="text-sm text-muted-foreground">Turn web search on/off for all modules</p>
                </div>
                <Switch
                  id="enabled"
                  checked={settings.enabled}
                  onCheckedChange={(checked) => setSettings({ ...settings, enabled: checked })}
                />
              </div>

              <div className="space-y-2">
                <Label>Maximum Results Per Query</Label>
                <Select
                  value={settings.maxResults.toString()}
                  onValueChange={(value) => setSettings({ ...settings, maxResults: parseInt(value) })}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="3">3 results</SelectItem>
                    <SelectItem value="5">5 results</SelectItem>
                    <SelectItem value="10">10 results</SelectItem>
                    <SelectItem value="20">20 results</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>Rate Limit (requests per minute)</Label>
                <Select
                  value={settings.rateLimit.toString()}
                  onValueChange={(value) => setSettings({ ...settings, rateLimit: parseInt(value) })}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="5">5 requests/min</SelectItem>
                    <SelectItem value="10">10 requests/min</SelectItem>
                    <SelectItem value="20">20 requests/min</SelectItem>
                    <SelectItem value="50">50 requests/min</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>

          {/* Safety Controls */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-2">
                <Shield className="h-5 w-5" />
                <CardTitle>Safety & Content Filtering</CardTitle>
              </div>
              <CardDescription>Control content safety and filtering</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="flex items-center justify-between">
                <div>
                  <Label htmlFor="safe-search" className="text-base">Safe Search</Label>
                  <p className="text-sm text-muted-foreground">Filter adult and unsafe content</p>
                </div>
                <Switch
                  id="safe-search"
                  checked={settings.safeSearch}
                  onCheckedChange={(checked) => setSettings({ ...settings, safeSearch: checked })}
                />
              </div>
            </CardContent>
          </Card>

          {/* Per-Module Controls */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-2">
                <Zap className="h-5 w-5" />
                <CardTitle>Module Permissions</CardTitle>
              </div>
              <CardDescription>Enable/disable web search for specific modules</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              {Object.entries(settings.modules).map(([module, enabled]) => (
                <div key={module} className="flex items-center justify-between">
                  <Label htmlFor={module} className="capitalize cursor-pointer">
                    Smart {module}
                  </Label>
                  <Switch
                    id={module}
                    checked={enabled}
                    onCheckedChange={(checked) =>
                      setSettings({
                        ...settings,
                        modules: { ...settings.modules, [module]: checked },
                      })
                    }
                  />
                </div>
              ))}
            </CardContent>
          </Card>

          {/* Save Button */}
          <div className="flex justify-end">
            <Button onClick={saveSettings} size="lg" className="gap-2">
              <Settings className="h-4 w-4" />
              Save Settings
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default StudioSearchSettings;
