import { WalletGuard } from "@/components/ide/WalletGuard";
import { Card } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Shield, Lock, AlertTriangle, Activity } from "lucide-react";
import { useState } from "react";
import { useToast } from "@/hooks/use-toast";

const StudioSecuritySettings = () => {
  const { toast } = useToast();
  const [settings, setSettings] = useState({
    twoFactor: false,
    ipRestriction: false,
    sessionTimeout: 30,
    loginAlerts: true,
    rateLimiting: true
  });

  const handleSave = () => {
    toast({
      title: "Security settings updated",
      description: "Your security preferences have been saved"
    });
  };

  return (
    <WalletGuard>
      <div className="min-h-screen bg-background p-6">
        <div className="max-w-4xl mx-auto">
          <div className="flex items-center gap-3 mb-6">
            <Shield className="h-8 w-8 text-primary" />
            <div>
              <h1 className="text-3xl font-bold text-foreground">Security Settings</h1>
              <p className="text-muted-foreground">Manage platform security and access control</p>
            </div>
          </div>

          <Tabs defaultValue="access">
            <TabsList>
              <TabsTrigger value="access">Access Control</TabsTrigger>
              <TabsTrigger value="logs">Security Logs</TabsTrigger>
              <TabsTrigger value="advanced">Advanced</TabsTrigger>
            </TabsList>

            <TabsContent value="access" className="mt-6 space-y-6">
              <Card className="p-6">
                <div className="flex items-center gap-2 mb-4">
                  <Lock className="h-5 w-5 text-primary" />
                  <h3 className="text-lg font-semibold">Authentication</h3>
                </div>

                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <Label htmlFor="two-factor">Two-Factor Authentication</Label>
                      <p className="text-sm text-muted-foreground">
                        Add an extra layer of security
                      </p>
                    </div>
                    <Switch
                      id="two-factor"
                      checked={settings.twoFactor}
                      onCheckedChange={(checked) =>
                        setSettings({ ...settings, twoFactor: checked })
                      }
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <div>
                      <Label htmlFor="login-alerts">Login Alerts</Label>
                      <p className="text-sm text-muted-foreground">
                        Get notified of new login attempts
                      </p>
                    </div>
                    <Switch
                      id="login-alerts"
                      checked={settings.loginAlerts}
                      onCheckedChange={(checked) =>
                        setSettings({ ...settings, loginAlerts: checked })
                      }
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="session-timeout">Session Timeout (minutes)</Label>
                    <Input
                      id="session-timeout"
                      type="number"
                      value={settings.sessionTimeout}
                      onChange={(e) =>
                        setSettings({ ...settings, sessionTimeout: parseInt(e.target.value) })
                      }
                    />
                  </div>
                </div>
              </Card>

              <Card className="p-6">
                <div className="flex items-center gap-2 mb-4">
                  <AlertTriangle className="h-5 w-5 text-primary" />
                  <h3 className="text-lg font-semibold">Rate Limiting</h3>
                </div>

                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <Label htmlFor="rate-limiting">API Rate Limiting</Label>
                      <p className="text-sm text-muted-foreground">
                        Protect against abuse and DDoS attacks
                      </p>
                    </div>
                    <Switch
                      id="rate-limiting"
                      checked={settings.rateLimiting}
                      onCheckedChange={(checked) =>
                        setSettings({ ...settings, rateLimiting: checked })
                      }
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <div>
                      <Label htmlFor="ip-restriction">IP Restriction</Label>
                      <p className="text-sm text-muted-foreground">
                        Limit access to specific IP addresses
                      </p>
                    </div>
                    <Switch
                      id="ip-restriction"
                      checked={settings.ipRestriction}
                      onCheckedChange={(checked) =>
                        setSettings({ ...settings, ipRestriction: checked })
                      }
                    />
                  </div>
                </div>
              </Card>

              <Button onClick={handleSave} className="w-full">
                Save Security Settings
              </Button>
            </TabsContent>

            <TabsContent value="logs" className="mt-6">
              <Card className="p-6">
                <div className="flex items-center gap-2 mb-4">
                  <Activity className="h-5 w-5 text-primary" />
                  <h3 className="text-lg font-semibold">Recent Security Events</h3>
                </div>

                <div className="space-y-3">
                  {[
                    { event: "Successful login", ip: "192.168.1.1", time: "2 minutes ago", status: "success" },
                    { event: "Failed login attempt", ip: "10.0.0.5", time: "1 hour ago", status: "warning" },
                    { event: "Password changed", ip: "192.168.1.1", time: "2 hours ago", status: "success" },
                    { event: "API rate limit exceeded", ip: "203.0.113.0", time: "3 hours ago", status: "warning" }
                  ].map((log, index) => (
                    <div key={index} className="flex items-center justify-between py-3 border-b">
                      <div>
                        <p className="font-medium">{log.event}</p>
                        <p className="text-sm text-muted-foreground">IP: {log.ip}</p>
                      </div>
                      <div className="text-right">
                        <span
                          className={`text-xs px-2 py-1 rounded ${
                            log.status === "success"
                              ? "bg-green-500/10 text-green-600"
                              : "bg-yellow-500/10 text-yellow-600"
                          }`}
                        >
                          {log.status}
                        </span>
                        <p className="text-xs text-muted-foreground mt-1">{log.time}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </Card>
            </TabsContent>

            <TabsContent value="advanced" className="mt-6">
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4">Advanced Security Options</h3>
                <p className="text-muted-foreground mb-4">
                  These settings are for advanced users only. Changing them incorrectly may affect platform security.
                </p>
                <Button variant="outline" className="w-full">
                  Configure Advanced Settings
                </Button>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </WalletGuard>
  );
};

export default StudioSecuritySettings;
