import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { ScrollArea } from "@/components/ui/scroll-area";
import { useTaskEngine, TaskType, TaskPriority } from "@/hooks/useTaskEngine";
import { useOwnerCheck } from "@/hooks/useOwnerCheck";
import { Play, Square, CheckCircle2, XCircle, Clock, AlertCircle, Sparkles } from "lucide-react";
import { formatDistanceToNow } from "date-fns";

export default function StudioTasks() {
  const { isOwner } = useOwnerCheck();
  const { 
    tasks, 
    runningTask, 
    createTask, 
    startTask, 
    cancelTask, 
    approveTask,
    getTaskLog,
    getTasksByStatus,
    getSuggestedTasks 
  } = useTaskEngine();

  const [newTask, setNewTask] = useState({
    title: "",
    description: "",
    type: "build_module" as TaskType,
    target: "",
    priority: "medium" as TaskPriority,
  });

  const handleCreateTask = () => {
    if (!newTask.title || !newTask.target) return;
    
    createTask(newTask);
    setNewTask({
      title: "",
      description: "",
      type: "build_module",
      target: "",
      priority: "medium",
    });
  };

  const handleStartTask = async (taskId: string) => {
    await startTask(taskId);
  };

  const pendingTasks = getTasksByStatus("pending");
  const runningTasks = getTasksByStatus("running");
  const completedTasks = getTasksByStatus("completed");
  const failedTasks = getTasksByStatus("failed");
  const suggestedTasks = getSuggestedTasks();

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "running": return <Clock className="h-4 w-4 animate-spin" />;
      case "completed": return <CheckCircle2 className="h-4 w-4 text-green-500" />;
      case "failed": return <XCircle className="h-4 w-4 text-red-500" />;
      case "cancelled": return <Square className="h-4 w-4 text-muted-foreground" />;
      default: return <Clock className="h-4 w-4" />;
    }
  };

  const getPriorityColor = (priority: TaskPriority) => {
    switch (priority) {
      case "critical": return "destructive";
      case "high": return "default";
      case "medium": return "secondary";
      case "low": return "outline";
      default: return "secondary";
    }
  };

  return (
    <div className="container mx-auto p-6 max-w-7xl">
      <div className="flex items-center justify-between mb-8">
        <div>
          <h1 className="text-4xl font-bold mb-2">AI Task Engine</h1>
          <p className="text-muted-foreground">Autonomous task management and execution</p>
        </div>
      </div>

      <Tabs defaultValue="all" className="space-y-6">
        <TabsList>
          <TabsTrigger value="all">All Tasks ({tasks.length})</TabsTrigger>
          <TabsTrigger value="pending">Pending ({pendingTasks.length})</TabsTrigger>
          <TabsTrigger value="running">Running ({runningTasks.length})</TabsTrigger>
          <TabsTrigger value="completed">Completed ({completedTasks.length})</TabsTrigger>
          <TabsTrigger value="failed">Failed ({failedTasks.length})</TabsTrigger>
          <TabsTrigger value="create">Create New</TabsTrigger>
          <TabsTrigger value="suggested">Suggested <Sparkles className="h-4 w-4 ml-1" /></TabsTrigger>
        </TabsList>

        <TabsContent value="all" className="space-y-4">
          {tasks.length === 0 ? (
            <Card>
              <CardContent className="flex flex-col items-center justify-center py-12">
                <AlertCircle className="h-12 w-12 text-muted-foreground mb-4" />
                <p className="text-muted-foreground">No tasks yet. Create your first task!</p>
              </CardContent>
            </Card>
          ) : (
            tasks.map(task => (
              <TaskCard 
                key={task.id} 
                task={task} 
                isOwner={isOwner}
                onStart={handleStartTask}
                onCancel={cancelTask}
                onApprove={approveTask}
                getStatusIcon={getStatusIcon}
                getPriorityColor={getPriorityColor}
                getTaskLog={getTaskLog}
                isRunning={runningTask === task.id}
              />
            ))
          )}
        </TabsContent>

        {["pending", "running", "completed", "failed"].map(status => (
          <TabsContent key={status} value={status} className="space-y-4">
            {getTasksByStatus(status as any).map(task => (
              <TaskCard 
                key={task.id} 
                task={task} 
                isOwner={isOwner}
                onStart={handleStartTask}
                onCancel={cancelTask}
                onApprove={approveTask}
                getStatusIcon={getStatusIcon}
                getPriorityColor={getPriorityColor}
                getTaskLog={getTaskLog}
                isRunning={runningTask === task.id}
              />
            ))}
          </TabsContent>
        ))}

        <TabsContent value="create">
          <Card>
            <CardHeader>
              <CardTitle>Create New Task</CardTitle>
              <CardDescription>Define a task for Smart Friend to execute</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="title">Title</Label>
                <Input
                  id="title"
                  value={newTask.title}
                  onChange={(e) => setNewTask({ ...newTask, title: e.target.value })}
                  placeholder="e.g., Build Smart Academy Exam Engine"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="description">Description</Label>
                <Textarea
                  id="description"
                  value={newTask.description}
                  onChange={(e) => setNewTask({ ...newTask, description: e.target.value })}
                  placeholder="Describe what needs to be done..."
                  rows={4}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="type">Task Type</Label>
                  <Select value={newTask.type} onValueChange={(value: TaskType) => setNewTask({ ...newTask, type: value })}>
                    <SelectTrigger id="type">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="build_module">Build Module</SelectItem>
                      <SelectItem value="fix_bug">Fix Bug</SelectItem>
                      <SelectItem value="refactor">Refactor</SelectItem>
                      <SelectItem value="deploy">Deploy</SelectItem>
                      <SelectItem value="optimize">Optimize</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="priority">Priority</Label>
                  <Select value={newTask.priority} onValueChange={(value: TaskPriority) => setNewTask({ ...newTask, priority: value })}>
                    <SelectTrigger id="priority">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="low">Low</SelectItem>
                      <SelectItem value="medium">Medium</SelectItem>
                      <SelectItem value="high">High</SelectItem>
                      <SelectItem value="critical">Critical</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="target">Target Project/Module</Label>
                <Input
                  id="target"
                  value={newTask.target}
                  onChange={(e) => setNewTask({ ...newTask, target: e.target.value })}
                  placeholder="e.g., smart-academy"
                />
              </div>

              <Button onClick={handleCreateTask} className="w-full">
                Create Task
              </Button>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="suggested">
          <div className="space-y-4">
            {suggestedTasks.map((suggested, index) => (
              <Card key={index}>
                <CardHeader>
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <CardTitle className="flex items-center gap-2">
                        <Sparkles className="h-5 w-5 text-primary" />
                        {suggested.title}
                      </CardTitle>
                      <CardDescription className="mt-2">{suggested.description}</CardDescription>
                    </div>
                    <Badge variant={getPriorityColor(suggested.priority)}>{suggested.priority}</Badge>
                  </div>
                </CardHeader>
                <CardContent>
                  <Button 
                    onClick={() => {
                      createTask({
                        title: suggested.title,
                        description: suggested.description,
                        type: suggested.type,
                        target: suggested.title.toLowerCase().replace(/\s+/g, '-'),
                        priority: suggested.priority,
                      });
                    }}
                  >
                    Create This Task
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>
      </Tabs>
    </div>
  );
}

function TaskCard({ task, isOwner, onStart, onCancel, onApprove, getStatusIcon, getPriorityColor, getTaskLog, isRunning }: any) {
  const [showLogs, setShowLogs] = useState(false);
  const logs = getTaskLog(task.id);
  const needsApproval = !task.ownerApproved && (task.priority === "critical" || task.type === "deploy");

  return (
    <Card>
      <CardHeader>
        <div className="flex items-start justify-between">
          <div className="flex-1">
            <div className="flex items-center gap-2 mb-2">
              {getStatusIcon(task.status)}
              <CardTitle>{task.title}</CardTitle>
            </div>
            <CardDescription>{task.description}</CardDescription>
          </div>
          <div className="flex gap-2">
            <Badge variant={getPriorityColor(task.priority)}>{task.priority}</Badge>
            <Badge variant="outline">{task.type.replace('_', ' ')}</Badge>
          </div>
        </div>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="flex items-center gap-4 text-sm text-muted-foreground">
          <span>Target: {task.target}</span>
          <span>•</span>
          <span>Created {formatDistanceToNow(task.createdAt, { addSuffix: true })}</span>
        </div>

        {needsApproval && task.status === "pending" && (
          <Card className="border-amber-500 bg-amber-50 dark:bg-amber-950">
            <CardContent className="pt-6">
              <p className="text-sm mb-3">⚠️ This task requires Owner approval before execution</p>
              {isOwner && (
                <Button onClick={() => onApprove(task.id)} variant="default" size="sm">
                  Approve as Owner (Engineer S M Nazrul Islam Tasnim)
                </Button>
              )}
            </CardContent>
          </Card>
        )}

        <div className="flex gap-2">
          {task.status === "pending" && (
            <Button 
              onClick={() => onStart(task.id)} 
              disabled={isRunning || (needsApproval && !task.ownerApproved)}
            >
              <Play className="h-4 w-4 mr-2" />
              Start Task
            </Button>
          )}
          
          {task.status === "running" && (
            <Button onClick={() => onCancel(task.id)} variant="destructive">
              <Square className="h-4 w-4 mr-2" />
              Cancel
            </Button>
          )}

          {logs.length > 0 && (
            <Button onClick={() => setShowLogs(!showLogs)} variant="outline">
              {showLogs ? "Hide" : "Show"} Logs ({logs.length})
            </Button>
          )}
        </div>

        {showLogs && logs.length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle className="text-sm">Task Logs</CardTitle>
            </CardHeader>
            <CardContent>
              <ScrollArea className="h-64">
                <div className="space-y-2">
                  {logs.map((log, index) => (
                    <div key={index} className="text-xs font-mono">
                      <span className="text-muted-foreground">
                        [{log.timestamp.toLocaleTimeString()}]
                      </span>
                      <span className={`ml-2 ${
                        log.level === "error" ? "text-red-500" :
                        log.level === "success" ? "text-green-500" :
                        log.level === "warning" ? "text-amber-500" :
                        "text-muted-foreground"
                      }`}>
                        [{log.level.toUpperCase()}]
                      </span>
                      <span className="ml-2">{log.message}</span>
                    </div>
                  ))}
                </div>
              </ScrollArea>
            </CardContent>
          </Card>
        )}

        {task.result && (
          <Card className={task.result.success ? "border-green-500" : "border-red-500"}>
            <CardContent className="pt-6">
              <p className="text-sm font-medium">
                {task.result.success ? "✓ " : "✗ "}
                {task.result.summary}
              </p>
            </CardContent>
          </Card>
        )}
      </CardContent>
    </Card>
  );
}
