import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

serve(async (req) => {
  const { headers } = req;
  const upgradeHeader = headers.get("upgrade") || "";

  if (upgradeHeader.toLowerCase() !== "websocket") {
    return new Response("Expected WebSocket connection", { status: 400 });
  }

  const { socket, response } = Deno.upgradeWebSocket(req);
  let openAISocket: WebSocket | null = null;
  let sessionToken = "";

  socket.onopen = () => {
    console.log("Client connected to AI Call Bot relay");
  };

  socket.onmessage = async (event) => {
    try {
      const message = JSON.parse(event.data);
      console.log("Received from client:", message.type);

      // Handle initial configuration
      if (message.type === "config" && message.token) {
        sessionToken = message.token;
        
        // Connect to OpenAI Realtime API
        const wsUrl = `wss://api.openai.com/v1/realtime?model=gpt-4o-realtime-preview-2024-12-17`;
        openAISocket = new WebSocket(wsUrl, {
          headers: {
            "Authorization": `Bearer ${sessionToken}`,
            "OpenAI-Beta": "realtime=v1"
          }
        });

        openAISocket.onopen = () => {
          console.log("Connected to OpenAI Realtime API");
          
          // Configure session
          openAISocket?.send(JSON.stringify({
            type: "session.update",
            session: {
              modalities: ["text", "audio"],
              instructions: "You are a helpful AI call assistant. Be professional and friendly.",
              voice: "alloy",
              input_audio_format: "pcm16",
              output_audio_format: "pcm16",
              input_audio_transcription: {
                model: "whisper-1"
              },
              turn_detection: {
                type: "server_vad",
                threshold: 0.5,
                prefix_padding_ms: 300,
                silence_duration_ms: 1000
              },
              temperature: 0.8,
              max_response_output_tokens: "inf"
            }
          }));
        };

        openAISocket.onmessage = (event) => {
          const aiMessage = JSON.parse(event.data);
          console.log("Received from OpenAI:", aiMessage.type);

          // Forward relevant events to client
          if (aiMessage.type === "response.audio.delta") {
            socket.send(JSON.stringify({
              type: "audio.delta",
              audio: aiMessage.delta
            }));
          } else if (aiMessage.type === "response.audio.done") {
            socket.send(JSON.stringify({
              type: "audio.done"
            }));
          } else if (aiMessage.type === "conversation.item.input_audio_transcription.completed") {
            socket.send(JSON.stringify({
              type: "transcript",
              speaker: "user",
              text: aiMessage.transcript
            }));
          } else if (aiMessage.type === "response.audio_transcript.delta") {
            socket.send(JSON.stringify({
              type: "transcript",
              speaker: "bot",
              text: aiMessage.delta
            }));
          }
        };

        openAISocket.onerror = (error) => {
          console.error("OpenAI WebSocket error:", error);
          socket.send(JSON.stringify({
            type: "error",
            message: "Connection to AI failed"
          }));
        };

        openAISocket.onclose = () => {
          console.log("OpenAI connection closed");
        };
      }

      // Forward audio input to OpenAI
      if (message.type === "audio.input" && openAISocket) {
        openAISocket.send(JSON.stringify({
          type: "input_audio_buffer.append",
          audio: message.audio
        }));
      }

    } catch (error) {
      console.error("Error handling message:", error);
    }
  };

  socket.onclose = () => {
    console.log("Client disconnected");
    openAISocket?.close();
  };

  socket.onerror = (error) => {
    console.error("Client WebSocket error:", error);
    openAISocket?.close();
  };

  return response;
});
