import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders });
  }

  try {
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_ANON_KEY') ?? '',
      { global: { headers: { Authorization: req.headers.get('Authorization')! } } }
    );

    const { data: { user }, error: authError } = await supabase.auth.getUser();
    if (authError || !user) {
      throw new Error('Unauthorized');
    }

    const { prompt, context, action } = await req.json();

    if (!prompt) {
      throw new Error('Prompt is required');
    }

    const LOVABLE_API_KEY = Deno.env.get('LOVABLE_API_KEY');
    if (!LOVABLE_API_KEY) {
      throw new Error('LOVABLE_API_KEY not configured');
    }

    let systemPrompt = '';
    
    switch (action) {
      case 'generate':
        systemPrompt = 'You are an expert code generator. Generate clean, efficient, production-ready code based on user requirements. Include proper error handling and comments.';
        break;
      case 'refactor':
        systemPrompt = 'You are an expert code refactorer. Analyze the code and suggest improvements for readability, performance, and maintainability. Preserve functionality while making it better.';
        break;
      case 'fix':
        systemPrompt = 'You are an expert code debugger. Analyze the code, identify issues, and provide fixes with clear explanations of what was wrong and how it was fixed.';
        break;
      case 'optimize':
        systemPrompt = 'You are an expert code optimizer. Analyze the code and optimize it for performance, memory usage, and best practices while maintaining functionality.';
        break;
      default:
        systemPrompt = 'You are an expert AI code assistant. Help users with their code-related tasks.';
    }

    const response = await fetch('https://ai.gateway.lovable.dev/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${LOVABLE_API_KEY}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        model: 'google/gemini-2.5-flash',
        messages: [
          { role: 'system', content: systemPrompt },
          { role: 'user', content: context ? `Context:\n${context}\n\nRequest: ${prompt}` : prompt }
        ],
      }),
    });

    if (!response.ok) {
      const error = await response.text();
      throw new Error(`AI Gateway error: ${error}`);
    }

    const data = await response.json();
    const generatedCode = data.choices[0]?.message?.content || '';

    // Log the action
    await supabase.from('core_brain_logs').insert({
      type: 'info',
      message: `AI Code Builder: ${action}`,
      details: {
        action,
        prompt: prompt.substring(0, 100),
        userId: user.id,
      },
    });

    return new Response(
      JSON.stringify({ 
        code: generatedCode,
        action,
        success: true,
      }),
      { headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );

  } catch (error) {
    console.error('AI Code Builder error:', error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : 'Unknown error' }),
      {
        status: 400,
        headers: { ...corsHeaders, 'Content-Type': 'application/json' },
      }
    );
  }
});
