import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { url, deepAnalysis = false } = await req.json();

    if (!url) {
      throw new Error("URL is required");
    }

    console.log("Analyzing website:", url, "Deep analysis:", deepAnalysis);

    // Fetch the website with proper headers
    const response = await fetch(url, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
      },
    });
    
    if (!response.ok) {
      throw new Error(`Failed to fetch: ${response.status}`);
    }
    
    const html = await response.text();

    // Enhanced analysis
    const analysis = {
      url,
      colors: extractColors(html),
      fonts: extractFonts(html),
      images: extractImages(html, url),
      meta: extractMetaTags(html),
      scripts: extractScripts(html),
      styles: extractStylesheets(html),
      layout: detectLayout(html),
      structure: deepAnalysis ? analyzeStructure(html) : null,
      improvements: generateImprovements(html),
      securityIssues: detectSecurityIssues(html),
    };

    console.log("Analysis complete:", {
      colors: analysis.colors.length,
      images: analysis.images.length,
      scripts: analysis.scripts.length,
    });

    return new Response(
      JSON.stringify(analysis),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (error) {
    console.error("Error in analyze-site:", error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});

function extractColors(html: string): string[] {
  const colorRegex = /#[0-9A-Fa-f]{6}|#[0-9A-Fa-f]{3}|rgb\([^)]+\)|rgba\([^)]+\)/g;
  const matches = html.match(colorRegex) || [];
  return Array.from(new Set(matches)).slice(0, 10);
}

function extractFonts(html: string): string[] {
  const fontRegex = /font-family:\s*([^;}"']+)/gi;
  const matches: string[] = [];
  let match;
  
  while ((match = fontRegex.exec(html)) !== null) {
    matches.push(match[1].trim());
  }
  
  return Array.from(new Set(matches)).slice(0, 5);
}

function extractImages(html: string, baseUrl: string): string[] {
  const imgRegex = /<img[^>]+src=["']([^"']+)["']/gi;
  const matches: string[] = [];
  let match;
  
  while ((match = imgRegex.exec(html)) !== null) {
    let src = match[1];
    if (src.startsWith('/')) {
      const base = new URL(baseUrl);
      src = `${base.origin}${src}`;
    } else if (!src.startsWith('http')) {
      src = new URL(src, baseUrl).href;
    }
    matches.push(src);
  }
  
  return matches.slice(0, 20);
}

function extractMetaTags(html: string): Record<string, string> {
  const meta: Record<string, string> = {};
  
  const titleMatch = html.match(/<title>([^<]+)<\/title>/i);
  if (titleMatch) meta.title = titleMatch[1];
  
  const descMatch = html.match(/<meta[^>]+name=["']description["'][^>]+content=["']([^"']+)["']/i);
  if (descMatch) meta.description = descMatch[1];
  
  const ogTitleMatch = html.match(/<meta[^>]+property=["']og:title["'][^>]+content=["']([^"']+)["']/i);
  if (ogTitleMatch) meta.ogTitle = ogTitleMatch[1];
  
  return meta;
}

function extractScripts(html: string): string[] {
  const scriptRegex = /<script[^>]+src=["']([^"']+)["']/gi;
  const matches: string[] = [];
  let match;
  
  while ((match = scriptRegex.exec(html)) !== null) {
    matches.push(match[1]);
  }
  
  return matches;
}

function extractStylesheets(html: string): string[] {
  const linkRegex = /<link[^>]+rel=["']stylesheet["'][^>]+href=["']([^"']+)["']/gi;
  const matches: string[] = [];
  let match;
  
  while ((match = linkRegex.exec(html)) !== null) {
    matches.push(match[1]);
  }
  
  return matches;
}

function detectLayout(html: string): string {
  if (html.includes('grid-template') || html.includes('display: grid')) {
    return "CSS Grid Layout";
  }
  if (html.includes('flex') || html.includes('display: flex')) {
    return "Flexbox Layout";
  }
  return "Traditional Layout";
}

function analyzeStructure(html: string): Record<string, boolean> {
  return {
    hasHeader: /<header/i.test(html) || /<nav/i.test(html),
    hasFooter: /<footer/i.test(html),
    hasHero: /hero|banner|jumbotron/i.test(html),
    hasSidebar: /sidebar|aside/i.test(html),
    hasNavigation: /<nav/i.test(html),
  };
}

function generateImprovements(html: string): string[] {
  const improvements: string[] = [];
  
  if (!html.includes('<meta name="description"')) {
    improvements.push("Add meta description for SEO");
  }
  
  if (!html.includes('alt=')) {
    improvements.push("Add alt text to images for accessibility");
  }
  
  if (html.includes('onclick=') || html.includes('onload=')) {
    improvements.push("Remove inline event handlers");
  }
  
  if (!html.includes('viewport')) {
    improvements.push("Add viewport meta tag for mobile");
  }
  
  return improvements;
}

function detectSecurityIssues(html: string): string[] {
  const issues: string[] = [];
  
  if (html.includes('eval(') || html.includes('innerHTML')) {
    issues.push("Potentially unsafe JavaScript detected");
  }
  
  if (html.match(/onclick|onload|onerror/gi)) {
    issues.push("Inline event handlers found (XSS risk)");
  }
  
  if (html.includes('googletagmanager') || html.includes('analytics')) {
    issues.push("Tracking scripts detected");
  }
  
  return issues;
}
