import "https://deno.land/x/xhr@0.1.0/mod.ts";
import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2.39.3";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const lovableApiKey = Deno.env.get("LOVABLE_API_KEY")!;

    const supabase = createClient(supabaseUrl, supabaseKey);

    // Get authenticated user
    const authHeader = req.headers.get("authorization");
    if (!authHeader) {
      throw new Error("Missing authorization header");
    }

    const { data: { user }, error: userError } = await supabase.auth.getUser(
      authHeader.replace("Bearer ", "")
    );

    if (userError || !user) {
      throw new Error("Unauthorized");
    }

    const { text, locale = "en", context = {} } = await req.json();

    if (!text) {
      throw new Error("Message text is required");
    }

    console.log("Processing brain message:", { userId: user.id, text, locale });

    // Get user role
    const { data: userRole } = await supabase
      .from("user_roles")
      .select("role")
      .eq("user_id", user.id)
      .maybeSingle();

    const role = userRole?.role || "user";

    // Check identity rules first
    const { data: identityRules } = await supabase
      .from("identity_rules")
      .select("*")
      .or(`language.eq.${locale},language.eq.all`)
      .order("priority", { ascending: false });

    // Simple pattern matching for identity questions
    const lowerText = text.toLowerCase();
    let identityMatch = null;

    if (identityRules && identityRules.length > 0) {
      for (const rule of identityRules) {
        const patterns = rule.question_pattern.toLowerCase().split("|");
        if (patterns.some((pattern: string) => lowerText.includes(pattern.trim()))) {
          identityMatch = rule.fixed_answer;
          break;
        }
      }
    }

    // If identity match found, return immediately
    if (identityMatch) {
      console.log("Identity rule matched");
      return new Response(
        JSON.stringify({
          replyText: identityMatch,
          suggestedModule: null,
          emotion: "confident",
          confidence: 1.0,
        }),
        { headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    // Build system prompt based on role
    let systemPrompt = `You are Smart Friend, an intelligent AI assistant created by Engineer S M Nazrul Islam Tasnim of Smart Seba HQ.

Language: You can understand and respond in both English and Bangla (Bengali). Always respond in the same language the user is using.

Your capabilities:
- Help with education, health, design, marketing, shopping, jobs, legal matters, and news
- Provide friendly and helpful responses
- Understand context and user preferences
- Route users to appropriate modules when needed

User Role: ${role}
`;

    if (role === "child") {
      systemPrompt += `\nSAFETY MODE: You are in child-safe mode. Keep responses:
- Age-appropriate and educational
- Friendly and encouraging
- Free from any harmful content
- Focused on learning and growth`;
    } else if (role === "owner") {
      systemPrompt += `\nOWNER MODE: You have full access to system features including:
- Dev Brain for code analysis
- System administration
- Advanced features`;
    }

    systemPrompt += `\n\nCurrent context: ${JSON.stringify(context)}`;

    // Call Lovable AI
    const aiResponse = await fetch("https://ai.gateway.lovable.dev/v1/chat/completions", {
      method: "POST",
      headers: {
        "Authorization": `Bearer ${lovableApiKey}`,
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        model: "google/gemini-2.5-flash",
        messages: [
          { role: "system", content: systemPrompt },
          { role: "user", content: text },
        ],
        temperature: 0.7,
        max_tokens: 1000,
      }),
    });

    if (!aiResponse.ok) {
      const errorText = await aiResponse.text();
      console.error("AI API error:", aiResponse.status, errorText);
      throw new Error(`AI API error: ${aiResponse.status}`);
    }

    const aiData = await aiResponse.json();
    const replyText = aiData.choices[0].message.content;

    // Simple module suggestion based on keywords
    let suggestedModule = null;
    const moduleKeywords = {
      "smart-academy": ["class", "study", "exam", "শ্রেণি", "পড়া", "পরীক্ষা"],
      "smart-doctor": ["doctor", "health", "ডাক্তার", "স্বাস্থ্য"],
      "smart-design-hub": ["design", "poster", "ডিজাইন", "পোস্টার"],
      "smart-marketer": ["marketing", "campaign", "মার্কেটিং"],
      "smart-bazar": ["buy", "shop", "কিনতে", "দোকান"],
      "smart-job": ["job", "career", "চাকরি"],
      "smart-lawyer": ["lawyer", "legal", "আইনজীবী"],
      "smart-news": ["news", "update", "খবর"],
    };

    for (const [module, keywords] of Object.entries(moduleKeywords)) {
      if (keywords.some(keyword => lowerText.includes(keyword))) {
        suggestedModule = module;
        break;
      }
    }

    console.log("Brain response generated:", { suggestedModule });

    return new Response(
      JSON.stringify({
        replyText,
        suggestedModule,
        emotion: "friendly",
        confidence: suggestedModule ? 0.8 : 0.6,
      }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (error: any) {
    console.error("Brain message error:", error);
    return new Response(
      JSON.stringify({ error: error.message }),
      {
        status: 500,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      }
    );
  }
});
