import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const LOVABLE_API_KEY = Deno.env.get('LOVABLE_API_KEY');
    const SUPABASE_URL = Deno.env.get('SUPABASE_URL');
    const SUPABASE_SERVICE_ROLE_KEY = Deno.env.get('SUPABASE_SERVICE_ROLE_KEY');

    if (!LOVABLE_API_KEY) {
      throw new Error('LOVABLE_API_KEY not configured');
    }

    const supabase = createClient(SUPABASE_URL!, SUPABASE_SERVICE_ROLE_KEY!);

    const { type, context } = await req.json();

    console.log('Core Brain Analyze request:', { type });

    let systemPrompt = '';
    let userPrompt = '';

    switch (type) {
      case 'error_analysis':
        systemPrompt = `You are an expert code analyst for Smart Friend app. Analyze error logs and provide:
1. Root cause analysis
2. Severity assessment (low/medium/high/critical)
3. Recommended fix approach
4. Estimated complexity
Format as JSON.`;
        userPrompt = `Analyze these error logs:\n${JSON.stringify(context.logs, null, 2)}`;
        break;

      case 'fix_suggestion':
        systemPrompt = `You are an expert software architect for Smart Friend app. Generate a detailed fix plan:
1. Step-by-step implementation plan
2. Files that need changes
3. Testing approach
4. Potential risks
Format as JSON with structured steps.`;
        userPrompt = `Generate fix plan for:\n${context.description}`;
        break;

      case 'code_patch':
        systemPrompt = `You are an expert code generator for Smart Friend app. Generate production-ready code patches:
1. Clean, well-commented code
2. Follow React/TypeScript best practices
3. Include error handling
4. Maintain existing code style
Return only the code patch, no explanations.`;
        userPrompt = `Generate code patch for:\nFile: ${context.filePath}\nTask: ${context.task}\nContext: ${context.existing || 'New file'}`;
        break;

      case 'module_scaffold':
        systemPrompt = `You are an expert full-stack developer for Smart Friend app. Generate COMPLETE, PRODUCTION-READY module scaffolding.

CRITICAL INSTRUCTIONS:
1. Return ONLY valid JSON (no markdown, no code blocks)
2. Include ALL necessary imports and proper TypeScript types
3. Follow Smart Friend's existing patterns (React, TypeScript, Tailwind, Supabase)
4. Generate working code that can be copied and used directly

Return JSON with this EXACT structure:
{
  "moduleName": "string",
  "description": "string",
  "features": ["string"],
  "routes": [
    {
      "path": "/module-path",
      "component": "ComponentName",
      "code": "// Route entry in App.tsx\n<Route path='/module-path' element={<ProtectedRoute><ComponentName /></ProtectedRoute>} />"
    }
  ],
  "components": [
    {
      "name": "ComponentName",
      "path": "src/pages/ComponentName.tsx",
      "code": "FULL COMPONENT CODE HERE"
    }
  ],
  "database": [
    {
      "schema": "table_name",
      "migrations": "COMPLETE SQL WITH RLS POLICIES"
    }
  ],
  "edgeFunctions": [
    {
      "name": "function-name",
      "code": "COMPLETE EDGE FUNCTION CODE"
    }
  ]
}

COMPONENT CODE REQUIREMENTS:
- Must have all necessary imports
- Use Smart Friend's design system (Card, Button, Badge, etc from @/components/ui)
- Include proper TypeScript types
- Add Supabase integration with proper error handling
- Responsive design with Tailwind
- Include toast notifications for actions
- Follow existing Smart Friend patterns

DATABASE REQUIREMENTS:
- Include CREATE TABLE with all columns
- Add RLS policies (enable RLS, SELECT, INSERT, UPDATE, DELETE)
- Use auth.uid() for user-specific data
- Include proper indexes
- Add triggers if needed

EDGE FUNCTION REQUIREMENTS:
- Include CORS headers
- Proper error handling
- Supabase client setup
- Input validation
- Logging`;
        userPrompt = `Generate complete module scaffold for:
Module Name: ${context.moduleName}
Description: ${context.description}
Features: ${context.features.join(', ')}

Generate 1 main page component, 1 database table with RLS, and 1 edge function.`;
        break;

      case 'test_generation':
        systemPrompt = `You are an expert test engineer. Generate comprehensive unit tests using Vitest and React Testing Library.
        
Return a JSON object with this exact structure:
{
  "test_code": "complete test code here",
  "framework": "vitest",
  "description": "brief description of what the test covers"
}

Guidelines:
- Use Vitest for the test framework
- Use @testing-library/react for React components
- Include describe, it/test blocks
- Test all edge cases and error scenarios
- Mock external dependencies (Supabase, APIs, etc.)
- Add proper assertions and expectations
- Follow testing best practices
- Make tests comprehensive but maintainable
- Include setup and teardown if needed`;
        
        userPrompt = `Generate unit tests to prevent this error from occurring again:

Error: ${context.error_message}
File: ${context.file_path || 'unknown'}
Issue Type: ${context.issue_type}
Severity: ${context.severity}

Additional Details: ${JSON.stringify(context.details)}

Create tests that:
1. Verify the issue doesn't happen again
2. Test edge cases that could cause similar errors
3. Validate error handling
4. Check boundary conditions
5. Mock external dependencies properly`;
        break;

      default:
        throw new Error('Invalid analysis type');
    }

    const aiResponse = await fetch('https://ai.gateway.lovable.dev/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${LOVABLE_API_KEY}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        model: 'google/gemini-2.5-flash',
        messages: [
          { role: 'system', content: systemPrompt },
          { role: 'user', content: userPrompt }
        ],
        temperature: 0.7,
      }),
    });

    if (!aiResponse.ok) {
      const errorText = await aiResponse.text();
      console.error('AI Gateway error:', aiResponse.status, errorText);
      
      if (aiResponse.status === 429) {
        return new Response(
          JSON.stringify({ error: 'Rate limit exceeded. Please try again later.' }),
          { status: 429, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
        );
      }
      
      if (aiResponse.status === 402) {
        return new Response(
          JSON.stringify({ error: 'Payment required. Please add credits to your Lovable AI workspace.' }),
          { status: 402, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
        );
      }

      throw new Error(`AI Gateway error: ${aiResponse.status}`);
    }

    const aiData = await aiResponse.json();
    const result = aiData.choices[0].message.content;

    console.log('AI Analysis complete:', type);

    return new Response(
      JSON.stringify({ success: true, result, type }),
      { headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );

  } catch (error) {
    console.error('Error in core-brain-analyze:', error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : 'Unknown error' }),
      { status: 500, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  }
});
