import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2.39.3";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

// Simulated file structure for Smart Friend project
const PROJECT_STRUCTURE = {
  frontend: [
    'src/pages/Auth.tsx',
    'src/pages/Chat.tsx',
    'src/pages/Contacts.tsx',
    'src/pages/CallHistory.tsx',
    'src/pages/Modules.tsx',
    'src/pages/Profile.tsx',
    'src/pages/CoreBrain.tsx',
    'src/components/BottomNav.tsx',
    'src/components/ProtectedRoute.tsx',
    'src/components/CoreBrainProtectedRoute.tsx',
    'src/hooks/useAuth.tsx',
    'src/hooks/useWebRTC.tsx',
  ],
  backend: [
    'supabase/functions/smart-friend-chat/index.ts',
    'supabase/functions/core-brain-analyze/index.ts',
    'supabase/functions/get-email-by-phone/index.ts',
  ]
};

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const SUPABASE_URL = Deno.env.get('SUPABASE_URL');
    const SUPABASE_SERVICE_ROLE_KEY = Deno.env.get('SUPABASE_SERVICE_ROLE_KEY');

    if (!SUPABASE_URL || !SUPABASE_SERVICE_ROLE_KEY) {
      throw new Error('Missing required environment variables');
    }

    const supabase = createClient(SUPABASE_URL, SUPABASE_SERVICE_ROLE_KEY);

    const { scanType, options } = await req.json();
    console.log('Scanner request:', { scanType, options });

    const results = [];

    // Scan error logs for patterns
    if (options.checkErrors) {
      const { data: logs } = await supabase
        .from('core_brain_logs')
        .select('*')
        .eq('type', 'error')
        .order('created_at', { ascending: false })
        .limit(10);

      if (logs && logs.length > 0) {
        // Pattern detection for common errors
        const errorPatterns = new Map<string, number>();
        logs.forEach(log => {
          const errorKey = log.message.substring(0, 50);
          errorPatterns.set(errorKey, (errorPatterns.get(errorKey) || 0) + 1);
        });

        errorPatterns.forEach((count, pattern) => {
          if (count > 1) {
            results.push({
              scan_type: scanType,
              file_path: null,
              issue_type: 'error',
              severity: count > 3 ? 'critical' : 'high',
              message: `Recurring error detected (${count}x): ${pattern}`,
              details: { count, pattern },
              resolved: false
            });
          }
        });
      }
    }

    // Check for performance issues
    if (options.checkPerformance) {
      // Simulate performance checks
      const performanceIssues = [
        {
          scan_type: scanType,
          file_path: 'src/pages/Chat.tsx',
          issue_type: 'performance',
          severity: 'medium',
          message: 'Large component re-renders detected - consider memoization',
          details: { 
            recommendation: 'Use React.memo() or useMemo() for expensive computations',
            lines: '150-200'
          },
          resolved: false
        },
        {
          scan_type: scanType,
          file_path: 'src/hooks/useWebRTC.tsx',
          issue_type: 'performance',
          severity: 'low',
          message: 'WebRTC connection cleanup may cause memory leaks',
          details: { 
            recommendation: 'Add proper cleanup in useEffect return',
            impact: 'Memory accumulation over time'
          },
          resolved: false
        }
      ];
      results.push(...performanceIssues);
    }

    // Check for unused code
    if (options.checkUnused) {
      const unusedIssues = [
        {
          scan_type: scanType,
          file_path: 'src/components/SmartFriendIcon.tsx',
          issue_type: 'unused',
          severity: 'low',
          message: 'Component defined but not imported anywhere',
          details: { 
            suggestion: 'Remove if not needed or import where needed'
          },
          resolved: false
        }
      ];
      results.push(...unusedIssues);
    }

    // Check for warnings
    if (options.checkWarnings) {
      const warningIssues = [
        {
          scan_type: scanType,
          file_path: 'src/pages/CoreBrain.tsx',
          issue_type: 'warning',
          severity: 'medium',
          message: 'Missing error boundary - unhandled errors may crash the app',
          details: { 
            recommendation: 'Wrap component tree with React Error Boundary',
            impact: 'Better error handling and user experience'
          },
          resolved: false
        },
        {
          scan_type: scanType,
          file_path: 'src/hooks/useAuth.tsx',
          issue_type: 'warning',
          severity: 'high',
          message: 'Auth state not persisted - users will be logged out on refresh',
          details: { 
            recommendation: 'Use Supabase auth session persistence',
            fix: 'Configure auth.storage in Supabase client'
          },
          resolved: false
        }
      ];
      results.push(...warningIssues);
    }

    // Database schema checks
    if (scanType === 'full' || scanType === 'database') {
      const { data: tables } = await supabase.rpc('is_core_brain_owner');
      
      // Check for missing indexes (simulated)
      results.push({
        scan_type: 'database',
        file_path: null,
        issue_type: 'performance',
        severity: 'medium',
        message: 'Missing index on messages.user_id - slow queries detected',
        details: { 
          table: 'messages',
          column: 'user_id',
          recommendation: 'CREATE INDEX idx_messages_user_id ON messages(user_id)'
        },
        resolved: false
      });

      // Check for RLS policy gaps
      results.push({
        scan_type: 'database',
        file_path: null,
        issue_type: 'warning',
        severity: 'high',
        message: 'Some tables lack comprehensive RLS policies',
        details: { 
          recommendation: 'Review and add missing UPDATE/DELETE policies'
        },
        resolved: false
      });
    }

    // Save results to database
    if (results.length > 0) {
      const { data: insertedResults, error: insertError } = await supabase
        .from('scan_results')
        .insert(results)
        .select();

      if (insertError) {
        console.error('Error saving scan results:', insertError);
      }

      // Track critical errors in notifications table
      if (insertedResults) {
        const criticalErrors = insertedResults.filter(r => r.severity === 'critical');
        if (criticalErrors.length > 0) {
          const notifications = criticalErrors.map(error => ({
            error_id: error.id,
            notification_type: 'in-app',
            details: {
              message: error.message,
              file_path: error.file_path,
              issue_type: error.issue_type,
              detected_at: new Date().toISOString()
            }
          }));

          const { error: notifError } = await supabase
            .from('critical_error_notifications')
            .insert(notifications);

          if (notifError) {
            console.error('Error saving notifications:', notifError);
          } else {
            console.log(`Created ${criticalErrors.length} critical error notifications`);
          }
        }
      }
    }

    // Log the scan activity
    await supabase.from('core_brain_logs').insert({
      type: 'info',
      message: `Project scan completed: ${results.length} issues found`,
      details: { scanType, issueCount: results.length, options }
    });

    return new Response(JSON.stringify({ 
      success: true,
      results,
      summary: {
        total: results.length,
        critical: results.filter(r => r.severity === 'critical').length,
        high: results.filter(r => r.severity === 'high').length,
        medium: results.filter(r => r.severity === 'medium').length,
        low: results.filter(r => r.severity === 'low').length,
      }
    }), {
      headers: { ...corsHeaders, 'Content-Type': 'application/json' },
    });

  } catch (error: any) {
    console.error('Error in core-brain-scanner:', error);
    return new Response(JSON.stringify({ error: error.message }), {
      status: 500,
      headers: { ...corsHeaders, 'Content-Type': 'application/json' },
    });
  }
});
