import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { functionName, arguments: args } = await req.json();
    
    const authHeader = req.headers.get("authorization");
    if (!authHeader) {
      throw new Error("No authorization header");
    }

    const supabaseClient = createClient(
      Deno.env.get("SUPABASE_URL") ?? "",
      Deno.env.get("SUPABASE_ANON_KEY") ?? "",
      { global: { headers: { Authorization: authHeader } } }
    );

    const { data: { user } } = await supabaseClient.auth.getUser();
    if (!user) {
      throw new Error("Not authenticated");
    }

    let result;

    switch (functionName) {
      case "set_reminder":
        result = await setReminder(supabaseClient, user.id, args);
        break;
      case "get_weather":
        result = await getWeather(args.location);
        break;
      case "search_web":
        result = await searchWeb(args.query);
        break;
      default:
        throw new Error(`Unknown function: ${functionName}`);
    }

    return new Response(JSON.stringify({ result }), {
      headers: { ...corsHeaders, "Content-Type": "application/json" },
    });
  } catch (error) {
    console.error("Error executing function:", error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});

async function setReminder(supabaseClient: any, userId: string, args: any) {
  const { title, description, remind_at } = args;
  
  const { data, error } = await supabaseClient
    .from("reminders")
    .insert({
      user_id: userId,
      title,
      description: description || null,
      remind_at: new Date(remind_at).toISOString(),
      is_completed: false,
    })
    .select()
    .single();

  if (error) {
    console.error("Error creating reminder:", error);
    throw new Error("Failed to create reminder");
  }

  return {
    success: true,
    message: `Reminder "${title}" set for ${new Date(remind_at).toLocaleString()}`,
    reminder: data,
  };
}

async function getWeather(location: string) {
  try {
    // Using wttr.in free weather API
    const response = await fetch(
      `https://wttr.in/${encodeURIComponent(location)}?format=j1`,
      {
        headers: {
          "User-Agent": "SmartFriend/1.0",
        },
      }
    );

    if (!response.ok) {
      throw new Error("Failed to fetch weather data");
    }

    const data = await response.json();
    const current = data.current_condition[0];
    const today = data.weather[0];

    return {
      location: data.nearest_area[0].areaName[0].value,
      country: data.nearest_area[0].country[0].value,
      temperature: `${current.temp_C}°C / ${current.temp_F}°F`,
      condition: current.weatherDesc[0].value,
      humidity: `${current.humidity}%`,
      wind_speed: `${current.windspeedKmph} km/h`,
      feels_like: `${current.FeelsLikeC}°C / ${current.FeelsLikeF}°F`,
      uv_index: current.uvIndex,
      forecast: {
        high: `${today.maxtempC}°C / ${today.maxtempF}°F`,
        low: `${today.mintempC}°C / ${today.mintempF}°F`,
      },
    };
  } catch (error) {
    console.error("Weather API error:", error);
    return {
      error: "Unable to fetch weather data for this location",
      location,
    };
  }
}

async function searchWeb(query: string) {
  try {
    // Using DuckDuckGo Instant Answer API (free, no API key needed)
    const response = await fetch(
      `https://api.duckduckgo.com/?q=${encodeURIComponent(query)}&format=json&no_html=1&skip_disambig=1`
    );

    if (!response.ok) {
      throw new Error("Failed to search");
    }

    const data = await response.json();
    
    // Collect results
    const results: any[] = [];
    
    if (data.AbstractText) {
      results.push({
        title: data.Heading || "Summary",
        snippet: data.AbstractText,
        url: data.AbstractURL,
        source: data.AbstractSource,
      });
    }

    if (data.RelatedTopics && data.RelatedTopics.length > 0) {
      data.RelatedTopics.slice(0, 3).forEach((topic: any) => {
        if (topic.Text && topic.FirstURL) {
          results.push({
            title: topic.Text.split(" - ")[0],
            snippet: topic.Text,
            url: topic.FirstURL,
          });
        }
      });
    }

    if (results.length === 0) {
      return {
        query,
        message: "No immediate results found. Try rephrasing your query.",
        results: [],
      };
    }

    return {
      query,
      results,
      total: results.length,
    };
  } catch (error) {
    console.error("Search API error:", error);
    return {
      error: "Unable to perform web search",
      query,
    };
  }
}
