import { serve } from "https://deno.land/std@0.168.0/http/server.ts";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders });
  }

  try {
    const projectFiles = {
      // Package.json with all dependencies
      'package.json': JSON.stringify({
        name: "smart-friend-production",
        private: true,
        version: "1.0.0",
        type: "module",
        scripts: {
          dev: "vite",
          build: "tsc && vite build",
          lint: "eslint . --ext ts,tsx",
          preview: "vite preview"
        },
        dependencies: {
          "@hookform/resolvers": "^3.10.0",
          "@monaco-editor/react": "^4.7.0",
          "@radix-ui/react-accordion": "^1.2.11",
          "@radix-ui/react-alert-dialog": "^1.1.14",
          "@radix-ui/react-avatar": "^1.1.10",
          "@radix-ui/react-checkbox": "^1.3.2",
          "@radix-ui/react-dialog": "^1.1.14",
          "@radix-ui/react-dropdown-menu": "^2.1.15",
          "@radix-ui/react-label": "^2.1.7",
          "@radix-ui/react-popover": "^1.1.14",
          "@radix-ui/react-scroll-area": "^1.2.9",
          "@radix-ui/react-select": "^2.2.5",
          "@radix-ui/react-separator": "^1.1.7",
          "@radix-ui/react-slot": "^1.2.3",
          "@radix-ui/react-switch": "^1.2.5",
          "@radix-ui/react-tabs": "^1.1.12",
          "@radix-ui/react-toast": "^1.2.14",
          "@radix-ui/react-tooltip": "^1.2.7",
          "@supabase/supabase-js": "^2.81.1",
          "@tanstack/react-query": "^5.83.0",
          "class-variance-authority": "^0.7.1",
          "clsx": "^2.1.1",
          "jszip": "^3.10.1",
          "lucide-react": "^0.462.0",
          "react": "^18.3.1",
          "react-dom": "^18.3.1",
          "react-hook-form": "^7.61.1",
          "react-router-dom": "^6.30.1",
          "sonner": "^1.7.4",
          "tailwind-merge": "^2.6.0",
          "tailwindcss-animate": "^1.0.7",
          "zod": "^3.25.76"
        },
        devDependencies: {
          "@types/node": "^20.11.0",
          "@types/react": "^18.3.1",
          "@types/react-dom": "^18.3.0",
          "@vitejs/plugin-react-swc": "^3.5.0",
          "autoprefixer": "^10.4.18",
          "postcss": "^8.4.35",
          "tailwindcss": "^3.4.1",
          "typescript": "^5.3.3",
          "vite": "^5.1.0"
        }
      }, null, 2),

      // Environment template
      '.env.example': `# Supabase Configuration (Update these with your production values)
VITE_SUPABASE_URL=https://akmuphwlcwvlkglpuvgj.supabase.co
VITE_SUPABASE_ANON_KEY=eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...
VITE_SUPABASE_PROJECT_ID=akmuphwlcwvlkglpuvgj

# Application
VITE_APP_NAME=Smart Friend
VITE_APP_VERSION=1.0.0`,

      // Deployment README
      'README-DEPLOY.md': `# Smart Friend - Complete Deployment Guide

## 📦 What's Included

This ZIP contains your COMPLETE Smart Friend project:
✅ All source code (pages, components, hooks)
✅ Supabase backend (functions, migrations)  
✅ Configuration files
✅ Public assets
✅ Build system ready

## 🚀 Quick Start (3 Steps)

### Step 1: Extract & Install
\`\`\`bash
# Extract the ZIP
unzip smart-friend-production.zip
cd smart-friend-production

# Install dependencies
npm install
\`\`\`

### Step 2: Configure Environment
\`\`\`bash
# Copy environment template
cp .env.example .env

# Edit .env with your Supabase credentials
# Get these from: https://supabase.com/dashboard/project/_/settings/api
\`\`\`

### Step 3: Build & Deploy
\`\`\`bash
# Build for production
npm run build

# Deploy the 'dist' folder to your hosting
\`\`\`

## 📤 cPanel Upload Instructions

### Method A: Direct Upload (Easiest)
1. Build locally: \`npm run build\`
2. Login to cPanel → File Manager
3. Navigate to \`public_html\`
4. Upload ALL files from the \`dist\` folder
5. Done! Visit your domain

### Method B: Full Project Upload (Advanced)
1. Upload entire ZIP to cPanel
2. Extract via File Manager
3. Open Terminal in cPanel
4. Run: \`npm install && npm run build\`
5. Copy dist contents to \`public_html\`

## ⚙️ Important: .htaccess File

Create \`.htaccess\` in \`public_html\` for React routing:

\`\`\`apache
<IfModule mod_rewrite.c>
  RewriteEngine On
  RewriteBase /
  RewriteRule ^index\\.html$ - [L]
  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteRule . /index.html [L]
</IfModule>
\`\`\`

## 🔐 Security Checklist

- [ ] Update .env with production Supabase credentials
- [ ] Enable HTTPS/SSL on your domain
- [ ] Configure CORS in Supabase dashboard for your domain
- [ ] Review RLS policies in Supabase

## 📁 Project Structure

\`\`\`
smart-friend-production/
├── src/              # Source code
├── supabase/         # Backend functions & migrations
├── public/           # Static assets
├── dist/             # Built files (after npm run build)
├── package.json      # Dependencies
├── vite.config.ts    # Build configuration
└── .env.example      # Environment template
\`\`\`

## 🆘 Troubleshooting

**Issue:** Blank page after deployment
→ Check .env has correct Supabase URL and key

**Issue:** 404 on page refresh  
→ Add .htaccess file (see above)

**Issue:** "npm not found" on cPanel
→ Build locally, upload only dist folder

## 📞 Support

- Check browser console for errors
- Review Supabase logs: https://supabase.com/dashboard/project/_/logs
- Verify all environment variables are set
`,

      // Vite config
      'vite.config.ts': `import { defineConfig } from "vite";
import react from "@vitejs/plugin-react-swc";
import path from "path";

export default defineConfig({
  server: {
    host: "::",
    port: 8080,
  },
  plugins: [react()],
  resolve: {
    alias: {
      "@": path.resolve(__dirname, "./src"),
    },
  },
  build: {
    outDir: "dist",
    sourcemap: false,
    minify: "esbuild",
  },
});`,

      // TypeScript config
      'tsconfig.json': JSON.stringify({
        compilerOptions: {
          target: "ES2020",
          useDefineForClassFields: true,
          lib: ["ES2020", "DOM", "DOM.Iterable"],
          module: "ESNext",
          skipLibCheck: true,
          moduleResolution: "bundler",
          allowImportingTsExtensions: true,
          resolveJsonModule: true,
          isolatedModules: true,
          noEmit: true,
          jsx: "react-jsx",
          strict: true,
          baseUrl: ".",
          paths: {
            "@/*": ["./src/*"]
          }
        },
        include: ["src"]
      }, null, 2),

      // Tailwind config
      'tailwind.config.ts': `import type { Config } from "tailwindcss";

export default {
  darkMode: ["class"],
  content: ["./src/**/*.{ts,tsx}"],
  theme: {
    extend: {
      colors: {
        border: "hsl(var(--border))",
        background: "hsl(var(--background))",
        foreground: "hsl(var(--foreground))",
        primary: {
          DEFAULT: "hsl(var(--primary))",
          foreground: "hsl(var(--primary-foreground))",
        },
      },
    },
  },
  plugins: [require("tailwindcss-animate")],
} satisfies Config;`,

      'postcss.config.js': `export default {
  plugins: {
    tailwindcss: {},
    autoprefixer: {},
  },
}`,

      '.gitignore': `node_modules
dist
.env
.env.local
*.log`,

      // Index HTML
      'index.html': `<!doctype html>
<html lang="en">
  <head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>Smart Friend</title>
  </head>
  <body>
    <div id="root"></div>
    <script type="module" src="/src/main.tsx"></script>
  </body>
</html>`,
    };

    // Return the manifest of files
    return new Response(
      JSON.stringify({
        success: true,
        files: Object.keys(projectFiles),
        totalFiles: Object.keys(projectFiles).length,
        message: "Full project export prepared. Note: This is a manifest. Full file reading requires repository access.",
        instructions: [
          "1. Download this project from Lovable using GitHub sync",
          "2. Clone the repository locally",
          "3. Run: npm install",
          "4. Run: npm run build",
          "5. Upload the 'dist' folder to your hosting"
        ]
      }),
      { 
        headers: { ...corsHeaders, 'Content-Type': 'application/json' }
      }
    );

  } catch (error) {
    console.error('Export error:', error);
    return new Response(
      JSON.stringify({ 
        error: error instanceof Error ? error.message : 'Export failed',
        suggestion: "Use GitHub integration to export full project"
      }),
      {
        status: 500,
        headers: { ...corsHeaders, 'Content-Type': 'application/json' },
      }
    );
  }
});
