import { createClient } from 'https://esm.sh/@supabase/supabase-js@2.81.1';

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

interface PaymentRequestAction {
  requestId: string;
  action: 'accept' | 'reject' | 'cancel';
}

Deno.serve(async (req) => {
  // Handle CORS preflight requests
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const supabaseUrl = Deno.env.get('SUPABASE_URL');
    const supabaseAnonKey = Deno.env.get('SUPABASE_ANON_KEY');
    const authHeader = req.headers.get('Authorization');

    if (!supabaseUrl || !supabaseAnonKey) {
      throw new Error('Missing required environment variables');
    }

    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return new Response(
        JSON.stringify({ error: 'Unauthorized' }),
        { status: 401, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    console.log('Initializing Supabase client...');
    const supabase = createClient(supabaseUrl, supabaseAnonKey, {
      auth: {
        persistSession: false,
      },
      global: {
        headers: { Authorization: authHeader },
      },
    });

    // Verify user authentication
    const {
      data: { user },
      error: authError,
    } = await supabase.auth.getUser();

    if (authError || !user) {
      console.error('Authentication failed:', authError);
      return new Response(
        JSON.stringify({ error: 'Unauthorized' }),
        { status: 401, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    console.log(`User authenticated: ${user.id}`);

    // Parse request body
    const { requestId, action }: PaymentRequestAction = await req.json();

    if (!requestId || !action) {
      return new Response(
        JSON.stringify({ error: 'Missing requestId or action' }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    console.log(`Processing ${action} for payment request: ${requestId}`);

    // Fetch the payment request
    const { data: paymentRequest, error: fetchError } = await supabase
      .from('payment_requests')
      .select('*')
      .eq('id', requestId)
      .single();

    if (fetchError || !paymentRequest) {
      console.error('Payment request not found:', fetchError);
      return new Response(
        JSON.stringify({ error: 'Payment request not found' }),
        { status: 404, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    console.log('Payment request found:', paymentRequest);

    // Verify user is authorized for this action
    if (action === 'cancel' && paymentRequest.requester_id !== user.id) {
      return new Response(
        JSON.stringify({ error: 'Only the requester can cancel this request' }),
        { status: 403, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    if ((action === 'accept' || action === 'reject') && paymentRequest.recipient_id !== user.id) {
      return new Response(
        JSON.stringify({ error: 'Only the recipient can accept or reject this request' }),
        { status: 403, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    // Check if request is still pending
    if (paymentRequest.status !== 'pending') {
      return new Response(
        JSON.stringify({ error: `Payment request is already ${paymentRequest.status}` }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    // Handle the action
    if (action === 'accept') {
      console.log('Processing payment transfer...');

      // Execute transfer using secure database function
      // Note: user.id is the payer (recipient of request), paymentRequest.requester_id gets the money
      const { data: transferResult, error: transferError } = await supabase.rpc(
        'execute_wallet_transfer',
        {
          _sender_id: user.id,
          _recipient_id: paymentRequest.requester_id,
          _amount: paymentRequest.amount,
          _note: `Payment request: ${paymentRequest.note || 'No note'}`
        }
      );

      if (transferError) {
        console.error('Transfer failed:', transferError);
        return new Response(
          JSON.stringify({ 
            error: transferError.message || 'Transfer failed'
          }),
          { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
        );
      }

      console.log('Payment transfer completed successfully');
    }

    // Update payment request status
    const newStatus = action === 'cancel' ? 'cancelled' : action === 'accept' ? 'accepted' : 'rejected';
    const { error: updateError } = await supabase
      .from('payment_requests')
      .update({ status: newStatus })
      .eq('id', requestId);

    if (updateError) {
      console.error('Failed to update payment request:', updateError);
      throw updateError;
    }

    // Log to core_brain_logs
    await supabase.from('core_brain_logs').insert({
      type: 'payment_request_action',
      message: `Payment request ${action}ed`,
      user_id: user.id,
      details: {
        request_id: requestId,
        action,
        amount: paymentRequest.amount,
      },
    });

    console.log(`Payment request ${action} successful`);

    return new Response(
      JSON.stringify({
        success: true,
        message: `Payment request ${action}ed successfully`,
      }),
      { status: 200, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  } catch (error) {
    console.error('Error handling payment request:', error);
    const errorMessage = error instanceof Error ? error.message : 'Internal server error';
    return new Response(
      JSON.stringify({ error: errorMessage }),
      { status: 500, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  }
});
