import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";
import { z } from "https://deno.land/x/zod@v3.22.4/mod.ts";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

// Input validation schema
const moduleGenRequestSchema = z.object({
  templateSlug: z.string().min(1).max(100).regex(/^[a-z0-9-]+$/, 'Invalid template slug format'),
  projectId: z.string().uuid('Invalid project ID'),
  customName: z.string().min(1).max(100).regex(/^[a-zA-Z0-9-_\s]+$/, 'Invalid custom name format').optional()
});

// Rate limiting: in-memory store (use Redis/Upstash in production)
const rateLimitStore = new Map<string, { count: number; resetAt: number }>();

function checkRateLimit(userId: string, maxRequests = 30): boolean {
  const now = Date.now();
  const userLimit = rateLimitStore.get(userId);
  
  if (!userLimit || now > userLimit.resetAt) {
    rateLimitStore.set(userId, { count: 1, resetAt: now + 60000 }); // 1 minute window
    return true;
  }
  
  if (userLimit.count >= maxRequests) {
    return false;
  }
  
  userLimit.count++;
  return true;
}

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const supabase = createClient(
      Deno.env.get("SUPABASE_URL") ?? "",
      Deno.env.get("SUPABASE_SERVICE_ROLE_KEY") ?? ""
    );

    // Get user
    const authHeader = req.headers.get("Authorization");
    if (!authHeader) {
      throw new Error("No authorization header");
    }

    const token = authHeader.replace("Bearer ", "");
    const { data: { user } } = await supabase.auth.getUser(token);
    if (!user) throw new Error("Not authenticated");

    // Rate limiting check
    if (!checkRateLimit(user.id)) {
      console.warn(`[Module Gen] Rate limit exceeded for user: ${user.id}`);
      return new Response(
        JSON.stringify({ error: 'Rate limit exceeded. Please try again later.' }),
        { 
          status: 429, 
          headers: { 
            ...corsHeaders, 
            'Content-Type': 'application/json',
            'Retry-After': '60'
          } 
        }
      );
    }

    // Parse and validate input
    const body = await req.json();
    const validationResult = moduleGenRequestSchema.safeParse(body);
    
    if (!validationResult.success) {
      console.error('[Module Gen] Validation failed:', validationResult.error.issues);
      return new Response(
        JSON.stringify({ 
          error: 'Invalid input',
          details: validationResult.error.issues.map(i => i.message).join(', ')
        }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    const { templateSlug, projectId, customName } = validationResult.data;

    console.log(`[Module Gen] User: ${user.id}, Template: ${templateSlug}, Project: ${projectId}`);

    // Get template
    const { data: template, error: templateError } = await supabase
      .from("module_templates")
      .select("*")
      .eq("slug", templateSlug)
      .single();

    if (templateError) throw templateError;

    const moduleName = customName || template.name;
    const config = template.template_config as any;

    // Generate folder structure
    const folders = [
      `/${moduleName}`,
      `/${moduleName}/src`,
      `/${moduleName}/src/controllers`,
      `/${moduleName}/src/models`,
      `/${moduleName}/src/routes`,
      `/${moduleName}/src/views`,
    ];

    const files = [
      {
        path: `/${moduleName}/README.md`,
        name: "README.md",
        content: `# ${moduleName}\n\n${template.description}\n\n## Features\n${config.features?.map((f: string) => `- ${f}`).join("\n") || ""}`,
      },
      {
        path: `/${moduleName}/module.json`,
        name: "module.json",
        content: JSON.stringify(template.template_config, null, 2),
      },
    ];

    // Laravel-specific files
    if (config.framework === "laravel") {
      files.push({
        path: `/${moduleName}/src/controllers/Controller.php`,
        name: "Controller.php",
        content: `<?php\n\nnamespace App\\Modules\\${moduleName}\\Controllers;\n\nuse Illuminate\\Http\\Request;\n\nclass Controller extends \\App\\Http\\Controllers\\Controller\n{\n    public function index()\n    {\n        return view('${moduleName}::index');\n    }\n}\n`,
      });

      files.push({
        path: `/${moduleName}/src/routes/web.php`,
        name: "web.php",
        content: `<?php\n\nuse Illuminate\\Support\\Facades\\Route;\nuse App\\Modules\\${moduleName}\\Controllers\\Controller;\n\nRoute::prefix('${templateSlug}')->group(function () {\n    Route::get('/', [Controller::class, 'index']);\n});\n`,
      });
    }

    // Create all folders and files
    for (const folder of folders) {
      await supabase.from("project_files").insert({
        user_id: user.id,
        project_id: projectId,
        file_path: folder,
        file_name: folder.split("/").pop(),
        file_type: "folder",
      });
    }

    for (const file of files) {
      await supabase.from("project_files").insert({
        user_id: user.id,
        project_id: projectId,
        file_path: file.path,
        file_name: file.name,
        file_type: "file",
        content: file.content,
      });
    }

    return new Response(
      JSON.stringify({ success: true, moduleName, filesCreated: files.length }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (error) {
    console.error("Error in module-generator:", error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
