import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { question, language = "bn", saveResponse = false } = await req.json();
    
    const authHeader = req.headers.get("authorization");
    if (!authHeader) {
      throw new Error("No authorization header");
    }

    const supabaseClient = createClient(
      Deno.env.get("SUPABASE_URL") ?? "",
      Deno.env.get("SUPABASE_ANON_KEY") ?? "",
      { global: { headers: { Authorization: authHeader } } }
    );

    // Search offline brain for matching response
    const { data: offlineResponses, error: searchError } = await supabaseClient
      .from("offline_brain")
      .select("*")
      .eq("language", language)
      .ilike("question", `%${question.toLowerCase()}%`)
      .limit(1);

    if (searchError) throw searchError;

    if (offlineResponses && offlineResponses.length > 0) {
      return new Response(
        JSON.stringify({
          answer: offlineResponses[0].answer,
          source: "offline_brain",
          category: offlineResponses[0].category,
        }),
        { headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    // If saveResponse is true and we don't have an answer, save it for future learning
    if (saveResponse) {
      const { data: { user } } = await supabaseClient.auth.getUser();
      if (user) {
        await supabaseClient.from("offline_brain").insert({
          question: question,
          answer: "", // Will be filled by AI later
          category: "general",
          language: language,
        });
      }
    }

    return new Response(
      JSON.stringify({
        answer: null,
        source: "none",
        message: "No offline response found",
      }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (error) {
    console.error("Error in offline-engine:", error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
