import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";
import { z } from "https://deno.land/x/zod@v3.22.4/mod.ts";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

// Input validation schema
const paymentRequestSchema = z.object({
  amount: z.number().positive().min(1).max(1000000),
  provider: z.enum(['bkash', 'nagad', 'stripe']),
  purpose: z.enum(['wallet', 'subscription', 'payout']),
  metadata: z.record(z.any()).optional()
});

// Rate limiting: in-memory store (use Redis/Upstash in production)
const rateLimitStore = new Map<string, { count: number; resetAt: number }>();

function checkRateLimit(userId: string, maxRequests = 10): boolean {
  const now = Date.now();
  const userLimit = rateLimitStore.get(userId);
  
  if (!userLimit || now > userLimit.resetAt) {
    rateLimitStore.set(userId, { count: 1, resetAt: now + 60000 }); // 1 minute window
    return true;
  }
  
  if (userLimit.count >= maxRequests) {
    return false;
  }
  
  userLimit.count++;
  return true;
}

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders });
  }

  try {
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_ANON_KEY') ?? '',
      { global: { headers: { Authorization: req.headers.get('Authorization')! } } }
    );

    const { data: { user }, error: authError } = await supabase.auth.getUser();
    if (authError || !user) {
      throw new Error('Unauthorized');
    }

    // Rate limiting check
    if (!checkRateLimit(user.id)) {
      console.warn(`[Payment Gateway] Rate limit exceeded for user: ${user.id}`);
      return new Response(
        JSON.stringify({ error: 'Rate limit exceeded. Please try again later.' }),
        { 
          status: 429, 
          headers: { 
            ...corsHeaders, 
            'Content-Type': 'application/json',
            'Retry-After': '60'
          } 
        }
      );
    }

    // Parse and validate input
    const body = await req.json();
    const validationResult = paymentRequestSchema.safeParse(body);
    
    if (!validationResult.success) {
      console.error('[Payment Gateway] Validation failed:', validationResult.error.issues);
      return new Response(
        JSON.stringify({ 
          error: 'Invalid input',
          details: validationResult.error.issues.map(i => i.message).join(', ')
        }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    const { amount, provider, purpose, metadata } = validationResult.data;

    console.log(`[Payment Gateway] User: ${user.id}, Provider: ${provider}, Amount: ${amount}`);

    // Generate payment URL based on provider
    let paymentUrl = '';
    let transactionId = `TXN_${Date.now()}_${user.id.substring(0, 8)}`;

    switch (provider) {
      case 'bkash':
        // bKash test mode
        paymentUrl = `https://checkout.sandbox.bkash.com/payment/create?amount=${amount}&invoice=${transactionId}`;
        break;
      
      case 'nagad':
        // Nagad test mode
        paymentUrl = `https://api.mynagad.com/api/dfs/check-out/initialize?amount=${amount}&orderId=${transactionId}`;
        break;
      
      case 'stripe':
        // Stripe test mode - would need Stripe SDK integration
        const STRIPE_SECRET_KEY = Deno.env.get('STRIPE_SECRET_KEY');
        if (!STRIPE_SECRET_KEY) {
          throw new Error('Stripe not configured');
        }
        
        const stripeResponse = await fetch('https://api.stripe.com/v1/checkout/sessions', {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${STRIPE_SECRET_KEY}`,
            'Content-Type': 'application/x-www-form-urlencoded',
          },
          body: new URLSearchParams({
            'payment_method_types[]': 'card',
            'line_items[0][price_data][currency]': 'usd',
            'line_items[0][price_data][product_data][name]': purpose,
            'line_items[0][price_data][unit_amount]': (amount * 100).toString(),
            'line_items[0][quantity]': '1',
            'mode': 'payment',
            'success_url': `${req.headers.get('origin')}/payment/success?session_id={CHECKOUT_SESSION_ID}`,
            'cancel_url': `${req.headers.get('origin')}/payment/cancel`,
          }),
        });

        if (!stripeResponse.ok) {
          throw new Error('Stripe payment creation failed');
        }

        const stripeData = await stripeResponse.json();
        paymentUrl = stripeData.url;
        transactionId = stripeData.id;
        break;

      default:
        throw new Error('Invalid payment provider');
    }

    // Log payment initiation
    await supabase.from('wallet_transactions').insert({
      user_id: user.id,
      type: 'credit',
      amount: amount,
      description: `${purpose} payment via ${provider} - ${transactionId}`,
    });

    return new Response(
      JSON.stringify({ 
        paymentUrl, 
        transactionId,
        provider,
        amount,
      }),
      { headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );

  } catch (error) {
    console.error('Payment gateway error:', error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : 'Unknown error' }),
      {
        status: 400,
        headers: { ...corsHeaders, 'Content-Type': 'application/json' },
      }
    );
  }
});
