import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

interface RateLimitConfig {
  maxRequests: number;
  windowMs: number;
}

const RATE_LIMITS: Record<string, RateLimitConfig> = {
  'chat': { maxRequests: 50, windowMs: 60000 }, // 50 requests per minute
  'voice': { maxRequests: 30, windowMs: 60000 }, // 30 requests per minute
  'search': { maxRequests: 100, windowMs: 60000 }, // 100 requests per minute
  'payment': { maxRequests: 10, windowMs: 60000 }, // 10 requests per minute
  'default': { maxRequests: 100, windowMs: 60000 } // 100 requests per minute
};

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders });
  }

  try {
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_SERVICE_ROLE_KEY') ?? ''
    );

    const { endpoint, userId, ipAddress } = await req.json();

    const identifier = userId || ipAddress || 'anonymous';
    const config = RATE_LIMITS[endpoint] || RATE_LIMITS.default;
    
    const now = Date.now();
    const windowStart = now - config.windowMs;
    
    // Store rate limit data in a simple table (you'd need to create this)
    // For now, we'll use in-memory storage with a simple counter
    // In production, use Redis or a dedicated rate limiting service
    
    const { data: limitData, error: fetchError } = await supabase
      .from('rate_limits')
      .select('*')
      .eq('identifier', identifier)
      .eq('endpoint', endpoint)
      .gte('timestamp', windowStart)
      .order('timestamp', { ascending: false });

    if (fetchError) {
      console.error('Rate limit fetch error:', fetchError);
    }

    const requestCount = limitData?.length || 0;

    if (requestCount >= config.maxRequests) {
      return new Response(
        JSON.stringify({ 
          error: 'Rate limit exceeded',
          message: `Too many requests. Please try again later.`,
          retryAfter: Math.ceil(config.windowMs / 1000)
        }),
        { 
          status: 429,
          headers: { 
            ...corsHeaders, 
            'Content-Type': 'application/json',
            'Retry-After': String(Math.ceil(config.windowMs / 1000))
          } 
        }
      );
    }

    // Log the request
    await supabase.from('rate_limits').insert({
      identifier,
      endpoint,
      timestamp: now
    });

    // Clean up old entries (optional, can be done via a scheduled job)
    await supabase
      .from('rate_limits')
      .delete()
      .lt('timestamp', windowStart);

    return new Response(
      JSON.stringify({ 
        allowed: true,
        remaining: config.maxRequests - requestCount - 1,
        resetAt: windowStart + config.windowMs
      }),
      { headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );

  } catch (error) {
    console.error('Rate limiter error:', error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : 'Unknown error' }),
      { 
        status: 500,
        headers: { ...corsHeaders, 'Content-Type': 'application/json' } 
      }
    );
  }
});
