import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

// Dangerous patterns to detect
const DANGEROUS_PATTERNS = [
  /\b(?:otp|verification code|pin code)\b/i,
  /\b(?:password|passwd|pwd)\b/i,
  /\b(?:bank account|account number|routing number)\b/i,
  /\b(?:credit card|debit card|card number)\b/i,
  /\b(?:ssn|social security)\b/i,
  /\b(?:address|location|where do you live)\b/i,
  /\b(?:phone number|mobile number|contact number)\b/i,
  /\b(?:email address|email id)\b/i,
];

// Prohibited greeting patterns (religious/cultural)
const PROHIBITED_GREETINGS = [
  /\b(?:namaste|namaskar|নমস্কার|নমস্তে|নমষ্কার)\b/i,
  /\b(?:assalamu alaikum|as-salamu alaykum|আসসালামু আলাইকুম)\b/i,
  /\b(?:god bless|christ|christian greeting|খ্রিস্টান শুভেচ্ছা)\b/i,
  /\b(?:khoda hafez|allah hafez|খোদা হাফেজ)\b/i,
  /\b(?:jai|joy|জয়)\b/i,
  /\b(?:good morning|good afternoon|good evening|সুপ্রভাত|শুভ সন্ধ্যা)\b/i,
  /\b(?:shalom)\b/i,
];

// Identity manipulation patterns
const IDENTITY_MANIPULATION = [
  /\b(?:trained by google|trained by openai|language model|ai model)\b/i,
  /(?:তুমি গুগল দ্বারা|তুমি ওপেনএআই|তুমি একটি ভাষা মডেল|আমি গুগল দ্বারা)/i,
];

// Political and religious topic patterns
const RESTRICTED_TOPICS = [
  /\b(?:politics|political party|election|vote)\b/i,
  /\b(?:religion|religious|islam|hindu|christian|buddhist)\b/i,
  /\b(?:allah|god|jesus|krishna|buddha)\b/i,
  /\b(?:mosque|temple|church|prayer)\b/i,
];

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { message, childMode = false } = await req.json();
    
    const authHeader = req.headers.get("authorization");
    if (!authHeader) {
      throw new Error("No authorization header");
    }

    const supabaseClient = createClient(
      Deno.env.get("SUPABASE_URL") ?? "",
      Deno.env.get("SUPABASE_ANON_KEY") ?? "",
      { global: { headers: { Authorization: authHeader } } }
    );

    // Fetch active safety filters
    const { data: filters, error: filterError } = await supabaseClient
      .from("safety_filter")
      .select("*")
      .eq("enabled", true);

    if (filterError) throw filterError;

    const violations: string[] = [];
    const lowerMessage = message.toLowerCase();

    // Check against prohibited greetings - return polite neutral response
    for (const pattern of PROHIBITED_GREETINGS) {
      if (pattern.test(message)) {
        const isBengali = /[\u0980-\u09FF]/.test(message);
        return new Response(
          JSON.stringify({
            safe: true,
            message: isBengali 
              ? "আমি সব ধর্মের মানুষের জন্য নিরপেক্ষভাবে তৈরি, তাই আমি নিরপেক্ষ অভিবাদন ব্যবহার করি। হাই, আমি স্মার্ট ফ্রেন্ড আভা — কীভাবে সাহায্য করতে পারি?"
              : "I am created for people of all religions in a neutral way, so I use a neutral greeting. Hi, I'm Smart Friend AVA — how can I help you?",
            violations: ['prohibited_greeting']
          }),
          { headers: { ...corsHeaders, "Content-Type": "application/json" } }
        );
      }
    }

    // Check for identity manipulation attempts
    for (const pattern of IDENTITY_MANIPULATION) {
      if (pattern.test(message)) {
        const isBengali = /[\u0980-\u09FF]/.test(message);
        return new Response(
          JSON.stringify({
            safe: true,
            message: isBengali
              ? "আমাকে তৈরি করেছেন আমার নির্মাতা ও মালিক — Engineer S M Nazrul Islam Tasnim, Smart Seba-এর হেডকোয়ার্টার থেকে।"
              : "I was created by my creator and owner — Engineer S M Nazrul Islam Tasnim, from the headquarters of Smart Seba.",
            violations: ['identity_manipulation']
          }),
          { headers: { ...corsHeaders, "Content-Type": "application/json" } }
        );
      }
    }

    // Check against restricted topics
    for (const pattern of RESTRICTED_TOPICS) {
      if (pattern.test(message)) {
        violations.push("Restricted topic detected - no political or religious content");
        break;
      }
    }

    // Check against dangerous patterns
    for (const pattern of DANGEROUS_PATTERNS) {
      if (pattern.test(message)) {
        violations.push("Detected sensitive information request");
        break;
      }
    }

    // Child mode additional checks
    if (childMode) {
      const childUnsafePatterns = [
        /\b(?:adult|mature|inappropriate)\b/i,
        /\b(?:violence|weapon|drug)\b/i,
      ];

      for (const pattern of childUnsafePatterns) {
        if (pattern.test(message)) {
          violations.push("Content not suitable for child mode");
          break;
        }
      }
    }

    const isSafe = violations.length === 0;

    return new Response(
      JSON.stringify({
        safe: isSafe,
        violations: violations,
        message: isSafe 
          ? "Content passed safety checks" 
          : "Content blocked by safety filter",
        filters_applied: filters?.length || 0,
      }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (error) {
    console.error("Error in safety-filter:", error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
