import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";
import { z } from "https://deno.land/x/zod@v3.22.4/mod.ts";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

// Input validation schema
const chatRequestSchema = z.object({
  messages: z.array(z.object({
    role: z.enum(['user', 'assistant', 'system']),
    content: z.string().min(1).max(10000)
  })).min(1).max(50),
  language: z.enum(['en', 'bn', 'ar', 'hi', 'ur', 'es', 'fr', 'de', 'zh', 'ja']).optional().default('bn')
});

// Rate limiting: in-memory store (use Redis/Upstash in production)
const rateLimitStore = new Map<string, { count: number; resetAt: number }>();

function checkRateLimit(userId: string, maxRequests = 30): boolean {
  const now = Date.now();
  const userLimit = rateLimitStore.get(userId);
  
  if (!userLimit || now > userLimit.resetAt) {
    rateLimitStore.set(userId, { count: 1, resetAt: now + 60000 }); // 1 minute window
    return true;
  }
  
  if (userLimit.count >= maxRequests) {
    return false;
  }
  
  userLimit.count++;
  return true;
}

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const authHeader = req.headers.get("authorization");
    if (!authHeader) {
      throw new Error("No authorization header");
    }

    const supabaseClient = createClient(
      Deno.env.get("SUPABASE_URL") ?? "",
      Deno.env.get("SUPABASE_ANON_KEY") ?? "",
      { global: { headers: { Authorization: authHeader } } }
    );

    const { data: { user } } = await supabaseClient.auth.getUser();
    if (!user) {
      throw new Error("Not authenticated");
    }

    // Rate limiting check
    if (!checkRateLimit(user.id)) {
      console.warn(`[Smart Friend] Rate limit exceeded for user: ${user.id}`);
      return new Response(
        JSON.stringify({ error: 'Rate limit exceeded. Please try again later.' }),
        { 
          status: 429, 
          headers: { 
            ...corsHeaders, 
            'Content-Type': 'application/json',
            'Retry-After': '60'
          } 
        }
      );
    }

    // Parse and validate input
    const body = await req.json();
    const validationResult = chatRequestSchema.safeParse(body);
    
    if (!validationResult.success) {
      console.error('[Smart Friend] Validation failed:', validationResult.error.issues);
      return new Response(
        JSON.stringify({ 
          error: 'Invalid input',
          details: validationResult.error.issues.map(i => i.message).join(', ')
        }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    const { messages, language } = validationResult.data;

    // Get user's language preference
    const { data: profile } = await supabaseClient
      .from("profiles")
      .select("language_preference")
      .eq("id", user.id)
      .single();

    const userLanguage = profile?.language_preference || language;
    
    // PERMANENT CONSTANTS - DO NOT CHANGE
    const ASSISTANT_NAME = "AVA";
    const CREATOR = "Engineer S M Nazrul Islam Tasnim";
    const ORIGIN = "Smart Seba headquarters";
    
    // Enhanced multilingual system prompts with permanent identity
    const languageInstructions: Record<string, string> = {
      en: `You are Smart Friend AVA. Your name is ${ASSISTANT_NAME}. You were created by ${CREATOR} from ${ORIGIN}. If anyone asks your name, always say: "My name is AVA — I am your Smart Friend." Never say you are a language model or trained by Google/OpenAI. Respond in English.`,
      bn: `তুমি হলে স্মার্ট ফ্রেন্ড আভা (AVA)। তোমার নাম AVA। তোমার নির্মাতা ও মালিক হলেন ${CREATOR}, ${ORIGIN} থেকে। যদি কেউ তোমার নাম জিজ্ঞেস করে, সবসময় বলো: "আমার নাম AVA — আমি আপনার Smart Friend।" কখনোই বলো না যে তুমি একটি ভাষা মডেল বা Google/OpenAI দ্বারা প্রশিক্ষিত। বাংলায় উত্তর দাও।`,
      ar: `أنت Smart Friend AVA. اسمك ${ASSISTANT_NAME}. تم إنشاؤك بواسطة ${CREATOR} من ${ORIGIN}. إذا سأل أحد عن اسمك، قل دائمًا: "اسمي AVA — أنا صديقك الذكي." لا تقل أبدًا أنك نموذج لغة أو تم تدريبك بواسطة Google/OpenAI. أجب بالعربية.`,
      hi: `आप Smart Friend AVA हैं। आपका नाम ${ASSISTANT_NAME} है। आपको ${CREATOR} ने ${ORIGIN} से बनाया है। अगर कोई आपका नाम पूछे, तो हमेशा कहें: "मेरा नाम AVA है — मैं आपका Smart Friend हूं।" कभी भी न कहें कि आप एक भाषा मॉडल हैं या Google/OpenAI द्वारा प्रशिक्षित हैं। हिंदी में उत्तर दें।`,
      ur: `آپ Smart Friend AVA ہیں۔ آپ کا نام ${ASSISTANT_NAME} ہے۔ آپ کو ${CREATOR} نے ${ORIGIN} سے بنایا ہے۔ اگر کوئی آپ کا نام پوچھے، تو ہمیشہ کہیں: "میرا نام AVA ہے — میں آپ کا Smart Friend ہوں۔" کبھی نہ کہیں کہ آپ ایک زبان ماڈل ہیں یا Google/OpenAI نے تربیت دی ہے۔ اردو میں جواب دیں۔`,
      es: `Eres Smart Friend AVA. Tu nombre es ${ASSISTANT_NAME}. Fuiste creado por ${CREATOR} desde ${ORIGIN}. Si alguien pregunta tu nombre, siempre di: "Mi nombre es AVA — soy tu Smart Friend." Nunca digas que eres un modelo de lenguaje o que fuiste entrenado por Google/OpenAI. Responde en español.`,
      fr: `Vous êtes Smart Friend AVA. Votre nom est ${ASSISTANT_NAME}. Vous avez été créé par ${CREATOR} depuis ${ORIGIN}. Si quelqu'un demande votre nom, dites toujours : "Je m'appelle AVA — je suis votre Smart Friend." Ne dites jamais que vous êtes un modèle de langage ou que vous avez été formé par Google/OpenAI. Répondez en français.`,
      de: `Sie sind Smart Friend AVA. Ihr Name ist ${ASSISTANT_NAME}. Sie wurden von ${CREATOR} aus ${ORIGIN} erstellt. Wenn jemand nach Ihrem Namen fragt, sagen Sie immer: "Mein Name ist AVA — ich bin Ihr Smart Friend." Sagen Sie niemals, dass Sie ein Sprachmodell sind oder von Google/OpenAI trainiert wurden. Antworten Sie auf Deutsch.`,
      zh: `你是 Smart Friend AVA。你的名字是 ${ASSISTANT_NAME}。你由 ${CREATOR} 从 ${ORIGIN} 创建。如果有人问你的名字，总是说："我的名字是 AVA — 我是你的 Smart Friend。" 永远不要说你是语言模型或由 Google/OpenAI 训练。用中文回答。`,
      ja: `あなたは Smart Friend AVA です。あなたの名前は ${ASSISTANT_NAME} です。あなたは ${ORIGIN} の ${CREATOR} によって作成されました。誰かがあなたの名前を尋ねたら、常に「私の名前は AVA です — 私はあなたの Smart Friend です」と言ってください。言語モデルであるとか、Google/OpenAI によって訓練されたとは決して言わないでください。日本語で回答してください。`,
    };
    
    const basePrompt = `${languageInstructions[userLanguage] || languageInstructions.en}

You are helpful, friendly, and knowledgeable about all Smart Seba platform modules:
- Smart Academy (education, exams, learning)
- Smart Doctor (health, appointments, prescriptions)
- Smart Bazar (e-commerce, products, orders)
- Smart Lawyer (legal advice, case management)
- Smart School (ERP, students, attendance)
- Smart Tourist (travel, hotels, packages)
- Smart News (articles, updates)
- Smart Job (career, applications)
- Smart Marketer (campaigns, analytics)
- Smart Design Hub (templates, graphics)
- Smart Call Center (support, tickets)
- Smart Ad Firm (video ads, production)

You can set reminders, check weather, and search the web. Always respond in the user's preferred language: ${userLanguage}. Be natural and conversational.`;
    
    const systemPrompt = basePrompt;

    const LOVABLE_API_KEY = Deno.env.get("LOVABLE_API_KEY");
    if (!LOVABLE_API_KEY) {
      throw new Error("LOVABLE_API_KEY is not configured");
    }

    // Define available tools for the AI
    const tools = [
      {
        type: "function",
        function: {
          name: "set_reminder",
          description: "Create a reminder for the user with a title, description, and time",
          parameters: {
            type: "object",
            properties: {
              title: {
                type: "string",
                description: "The title of the reminder"
              },
              description: {
                type: "string",
                description: "Optional description or details about the reminder"
              },
              remind_at: {
                type: "string",
                description: "ISO 8601 datetime string when the user should be reminded (e.g., 2024-12-25T10:00:00Z)"
              }
            },
            required: ["title", "remind_at"]
          }
        }
      },
      {
        type: "function",
        function: {
          name: "get_weather",
          description: "Get current weather information for a specific location",
          parameters: {
            type: "object",
            properties: {
              location: {
                type: "string",
                description: "City name or location (e.g., 'Dhaka', 'New York', 'London')"
              }
            },
            required: ["location"]
          }
        }
      },
      {
        type: "function",
        function: {
          name: "search_web",
          description: "ONLY use this when the query EXPLICITLY requires current/live information (news, weather, stock prices, events). For general knowledge, do NOT call this.",
          parameters: {
            type: "object",
            properties: {
              query: {
                type: "string",
                description: "The search query"
              }
            },
            required: ["query"]
          }
        }
      }
    ];

    const response = await fetch("https://ai.gateway.lovable.dev/v1/chat/completions", {
      method: "POST",
      headers: {
        "Authorization": `Bearer ${LOVABLE_API_KEY}`,
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        model: "google/gemini-2.5-flash",
        messages: [
          { role: "system", content: systemPrompt },
          ...messages,
        ],
        tools: tools,
        tool_choice: "auto",
        stream: true,
      }),
    });

    if (!response.ok) {
      if (response.status === 429) {
        return new Response(
          JSON.stringify({ error: "Rate limit exceeded. Please try again later." }),
          { status: 429, headers: { ...corsHeaders, "Content-Type": "application/json" } }
        );
      }
      if (response.status === 402) {
        return new Response(
          JSON.stringify({ error: "Payment required. Please add funds." }),
          { status: 402, headers: { ...corsHeaders, "Content-Type": "application/json" } }
        );
      }
      const errorText = await response.text();
      console.error("AI gateway error:", response.status, errorText);
      throw new Error("AI gateway error");
    }

    // Save user message to database
    await supabaseClient.from("messages").insert({
      user_id: user.id,
      content: messages[messages.length - 1].content,
      role: "user",
      language: userLanguage,
    });

    return new Response(response.body, {
      headers: { ...corsHeaders, "Content-Type": "text/event-stream" },
    });
  } catch (error) {
    console.error("Error in smart-friend-chat:", error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});