import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2.39.3";
import { z } from "https://deno.land/x/zod@v3.22.4/mod.ts";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

// Input validation schema
const transferRequestSchema = z.object({
  recipientPhone: z.string().min(10).max(15).regex(/^[0-9+\s-]+$/, 'Invalid phone number format'),
  amount: z.number().positive().min(1).max(1000000),
  note: z.string().max(500).optional()
});

// Rate limiting: in-memory store (use Redis/Upstash in production)
const rateLimitStore = new Map<string, { count: number; resetAt: number }>();

function checkRateLimit(userId: string, maxRequests = 10): boolean {
  const now = Date.now();
  const userLimit = rateLimitStore.get(userId);
  
  if (!userLimit || now > userLimit.resetAt) {
    rateLimitStore.set(userId, { count: 1, resetAt: now + 60000 }); // 1 minute window
    return true;
  }
  
  if (userLimit.count >= maxRequests) {
    return false;
  }
  
  userLimit.count++;
  return true;
}

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const SUPABASE_URL = Deno.env.get('SUPABASE_URL');
    const SUPABASE_ANON_KEY = Deno.env.get('SUPABASE_ANON_KEY');

    if (!SUPABASE_URL || !SUPABASE_ANON_KEY) {
      throw new Error('Missing required environment variables');
    }

    // Get authenticated user from Authorization header
    const authHeader = req.headers.get('Authorization');
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return new Response(
        JSON.stringify({ success: false, error: 'Unauthorized' }),
        { status: 401, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    const token = authHeader.replace('Bearer ', '');

    const supabase = createClient(SUPABASE_URL, SUPABASE_ANON_KEY, {
      global: {
        headers: {
          Authorization: `Bearer ${token}`,
        },
      },
    });

    const { data: { user }, error: authError } = await supabase.auth.getUser();

    if (authError || !user) {
      throw new Error('Unauthorized');
    }

    // Rate limiting check
    if (!checkRateLimit(user.id)) {
      console.warn(`[Smart Pay] Rate limit exceeded for user: ${user.id}`);
      return new Response(
        JSON.stringify({ 
          success: false, 
          error: 'Rate limit exceeded. Please try again later.' 
        }),
        { 
          status: 429, 
          headers: { 
            ...corsHeaders, 
            'Content-Type': 'application/json',
            'Retry-After': '60'
          } 
        }
      );
    }

    // Parse and validate input
    const body = await req.json();
    const validationResult = transferRequestSchema.safeParse(body);
    
    if (!validationResult.success) {
      console.error('[Smart Pay] Validation failed:', validationResult.error.issues);
      return new Response(
        JSON.stringify({ 
          success: false,
          error: 'Invalid input',
          details: validationResult.error.issues.map(i => i.message).join(', ')
        }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    const { recipientPhone, amount, note } = validationResult.data;

    console.log('[Smart Pay] Transfer request:', { recipientPhone, amount, userId: user.id });

    // Find recipient by phone number
    const { data: recipientProfile, error: profileError } = await supabase
      .from('profiles')
      .select('id, full_name')
      .eq('phone', recipientPhone)
      .maybeSingle();

    if (profileError) {
      console.error('Error finding recipient:', profileError);
      throw new Error('Failed to find recipient');
    }

    if (!recipientProfile) {
      return new Response(
        JSON.stringify({ success: false, error: 'Recipient not found. Please check the phone number.' }),
        { headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    if (recipientProfile.id === user.id) {
      return new Response(
        JSON.stringify({ success: false, error: 'Cannot send money to yourself' }),
        { headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    // Execute transfer using secure database function
    // This function enforces authorization and handles transaction atomically
    const { data: transferResult, error: transferError } = await supabase.rpc(
      'execute_wallet_transfer',
      {
        _sender_id: user.id,
        _recipient_id: recipientProfile.id,
        _amount: amount,
        _note: note || null
      }
    );

    if (transferError) {
      console.error('Transfer failed:', transferError);
      return new Response(
        JSON.stringify({ 
          success: false, 
          error: transferError.message || 'Transfer failed'
        }),
        { headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    return new Response(
      JSON.stringify({
        success: true,
        message: `Transferred ৳${amount} to ${recipientProfile.full_name}`,
        transaction: {
          amount,
          recipient: recipientProfile.full_name
        }
      }),
      { headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );

  } catch (error: any) {
    console.error('Error in smart-pay-transfer:', error);
    return new Response(
      JSON.stringify({ success: false, error: error.message }),
      { status: 500, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  }
});
