import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders });
  }

  try {
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_ANON_KEY') ?? '',
      { global: { headers: { Authorization: req.headers.get('Authorization')! } } }
    );

    const { data: { user }, error: authError } = await supabase.auth.getUser();
    if (authError || !user) {
      throw new Error('Unauthorized');
    }

    const { transactionId } = await req.json();

    if (!transactionId) {
      throw new Error('Transaction ID is required');
    }

    // In production, you would verify with the actual payment provider
    // For now, we'll simulate verification
    const isVerified = true;
    const status = isVerified ? 'success' : 'failed';

    if (isVerified) {
      // Update wallet balance
      const { data: wallet } = await supabase
        .from('smart_wallets')
        .select('balance')
        .eq('user_id', user.id)
        .single();

      if (wallet) {
        // Get transaction amount from description
        const { data: transaction } = await supabase
          .from('wallet_transactions')
          .select('amount')
          .eq('user_id', user.id)
          .ilike('description', `%${transactionId}%`)
          .single();

        if (transaction) {
          await supabase
            .from('smart_wallets')
            .update({ balance: wallet.balance + transaction.amount })
            .eq('user_id', user.id);
        }
      }
    }

    return new Response(
      JSON.stringify({ 
        status,
        transactionId,
        verified: isVerified,
      }),
      { headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );

  } catch (error) {
    console.error('Payment verification error:', error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : 'Unknown error' }),
      {
        status: 400,
        headers: { ...corsHeaders, 'Content-Type': 'application/json' },
      }
    );
  }
});
