import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

// Rate limiting: simple in-memory store (for demo - use Redis in production)
const rateLimitStore = new Map<string, { count: number; resetAt: number }>();

function checkRateLimit(userId: string, maxRequests = 10): boolean {
  const now = Date.now();
  const userLimit = rateLimitStore.get(userId);
  
  if (!userLimit || now > userLimit.resetAt) {
    rateLimitStore.set(userId, { count: 1, resetAt: now + 60000 }); // 1 minute window
    return true;
  }
  
  if (userLimit.count >= maxRequests) {
    return false;
  }
  
  userLimit.count++;
  return true;
}

// Mock web search function - replace with real API later
// Supports: Tavily, SerpAPI, Brave Search, or OpenAI browsing
async function runWebSearch(query: string, options: {
  lang?: string;
  maxResults?: number;
  module?: string;
  safeSearch?: boolean;
}): Promise<any[]> {
  const TAVILY_API_KEY = Deno.env.get("TAVILY_API_KEY");
  
  if (!TAVILY_API_KEY) {
    console.warn('[Web Search] TAVILY_API_KEY not configured, using mock data');
    // Mock results fallback
    const mockResults = [
      {
        title: `${query} - Comprehensive Guide`,
        url: `https://example.com/search?q=${encodeURIComponent(query)}`,
        snippet: `Learn everything about ${query}. This is a comprehensive guide covering all aspects...`,
        source: "example.com",
        score: 0.95,
        faviconUrl: "https://example.com/favicon.ico",
        publishedDate: new Date().toISOString(),
      },
      {
        title: `Top 10 ${query} Tips`,
        url: `https://wikipedia.org/wiki/${encodeURIComponent(query)}`,
        snippet: `Discover the best practices and tips for ${query}. Expert recommendations and insights...`,
        source: "wikipedia.org",
        score: 0.88,
        faviconUrl: "https://wikipedia.org/favicon.ico",
        publishedDate: new Date(Date.now() - 86400000).toISOString(),
      },
    ];
    return mockResults.slice(0, options.maxResults || 5);
  }

  try {
    console.log(`[Web Search] Using Tavily API for query: "${query}"`);
    
    const response = await fetch("https://api.tavily.com/search", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({
        api_key: TAVILY_API_KEY,
        query,
        max_results: options.maxResults || 5,
        search_depth: "basic",
        include_answer: true,
        include_images: false,
        include_raw_content: false,
      })
    });

    if (!response.ok) {
      throw new Error(`Tavily API error: ${response.status}`);
    }

    const data = await response.json();
    
    return data.results.map((result: any) => ({
      title: result.title,
      url: result.url,
      snippet: result.content,
      source: new URL(result.url).hostname,
      score: result.score,
      publishedDate: result.published_date || new Date().toISOString(),
    }));
  } catch (error) {
    console.error('[Web Search] Tavily API error:', error);
    throw error;
  }
}

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const authHeader = req.headers.get("authorization");
    if (!authHeader) {
      throw new Error("No authorization header");
    }

    const supabaseClient = createClient(
      Deno.env.get("SUPABASE_URL") ?? "",
      Deno.env.get("SUPABASE_ANON_KEY") ?? "",
      { global: { headers: { Authorization: authHeader } } }
    );

    const { data: { user } } = await supabaseClient.auth.getUser();
    if (!user) {
      throw new Error("Not authenticated");
    }

    // Rate limiting check
    if (!checkRateLimit(user.id, 10)) {
      return new Response(
        JSON.stringify({ error: "Rate limit exceeded. Please try again later." }),
        { status: 429, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    const { query, lang = "bn", maxResults = 5, module, safeSearch = true } = await req.json();
    
    if (!query || query.trim().length === 0) {
      return new Response(
        JSON.stringify({ error: "Query is required" }),
        { status: 400, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    const startTime = Date.now();
    
    // Run the search
    const results = await runWebSearch(query, { lang, maxResults, module, safeSearch });
    
    const tookMs = Date.now() - startTime;

    // Log the search for analytics
    await supabaseClient.from("core_brain_logs").insert({
      type: "web_search",
      message: `Web search: "${query}"`,
      details: {
        query,
        lang,
        module,
        resultsCount: results.length,
        tookMs,
        userId: user.id,
      },
    });

    const response = {
      results,
      query,
      tookMs,
      lang,
      module,
    };

    return new Response(JSON.stringify(response), {
      headers: { ...corsHeaders, "Content-Type": "application/json" },
    });
  } catch (error) {
    console.error("Error in web-search:", error);
    return new Response(
      JSON.stringify({ error: error instanceof Error ? error.message : "Unknown error" }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
