-- Create app_role enum for user roles
CREATE TYPE public.app_role AS ENUM ('admin', 'user');

-- Create profiles table
CREATE TABLE public.profiles (
  id UUID PRIMARY KEY REFERENCES auth.users(id) ON DELETE CASCADE,
  full_name TEXT NOT NULL,
  phone TEXT,
  language_preference TEXT DEFAULT 'bn',
  avatar_url TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- Create user_roles table for role-based access control
CREATE TABLE public.user_roles (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE NOT NULL,
  role app_role NOT NULL,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  UNIQUE(user_id, role)
);

-- Create messages table for chat history
CREATE TABLE public.messages (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE NOT NULL,
  content TEXT NOT NULL,
  role TEXT NOT NULL CHECK (role IN ('user', 'assistant')),
  language TEXT DEFAULT 'bn',
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- Create modules table
CREATE TABLE public.modules (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  name_bn TEXT NOT NULL,
  description TEXT,
  description_bn TEXT,
  icon TEXT,
  route TEXT NOT NULL,
  is_active BOOLEAN DEFAULT true,
  order_index INTEGER DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- Create reminders table
CREATE TABLE public.reminders (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE NOT NULL,
  title TEXT NOT NULL,
  description TEXT,
  remind_at TIMESTAMPTZ NOT NULL,
  is_completed BOOLEAN DEFAULT false,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- Enable RLS
ALTER TABLE public.profiles ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.user_roles ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.messages ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.modules ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.reminders ENABLE ROW LEVEL SECURITY;

-- Create security definer function to check roles
CREATE OR REPLACE FUNCTION public.has_role(_user_id UUID, _role app_role)
RETURNS BOOLEAN
LANGUAGE SQL
STABLE
SECURITY DEFINER
SET search_path = public
AS $$
  SELECT EXISTS (
    SELECT 1
    FROM public.user_roles
    WHERE user_id = _user_id AND role = _role
  )
$$;

-- RLS Policies for profiles
CREATE POLICY "Users can view their own profile"
  ON public.profiles FOR SELECT
  USING (auth.uid() = id);

CREATE POLICY "Users can update their own profile"
  ON public.profiles FOR UPDATE
  USING (auth.uid() = id);

CREATE POLICY "Users can insert their own profile"
  ON public.profiles FOR INSERT
  WITH CHECK (auth.uid() = id);

-- RLS Policies for user_roles
CREATE POLICY "Admins can view all roles"
  ON public.user_roles FOR SELECT
  USING (public.has_role(auth.uid(), 'admin'));

CREATE POLICY "Users can view their own roles"
  ON public.user_roles FOR SELECT
  USING (auth.uid() = user_id);

CREATE POLICY "Admins can manage roles"
  ON public.user_roles FOR ALL
  USING (public.has_role(auth.uid(), 'admin'));

-- RLS Policies for messages
CREATE POLICY "Users can view their own messages"
  ON public.messages FOR SELECT
  USING (auth.uid() = user_id);

CREATE POLICY "Users can insert their own messages"
  ON public.messages FOR INSERT
  WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can delete their own messages"
  ON public.messages FOR DELETE
  USING (auth.uid() = user_id);

-- RLS Policies for modules
CREATE POLICY "Anyone can view active modules"
  ON public.modules FOR SELECT
  USING (is_active = true);

CREATE POLICY "Admins can manage modules"
  ON public.modules FOR ALL
  USING (public.has_role(auth.uid(), 'admin'));

-- RLS Policies for reminders
CREATE POLICY "Users can view their own reminders"
  ON public.reminders FOR SELECT
  USING (auth.uid() = user_id);

CREATE POLICY "Users can manage their own reminders"
  ON public.reminders FOR ALL
  USING (auth.uid() = user_id);

-- Function to handle new user signup
CREATE OR REPLACE FUNCTION public.handle_new_user()
RETURNS TRIGGER
LANGUAGE plpgsql
SECURITY DEFINER
SET search_path = public
AS $$
BEGIN
  INSERT INTO public.profiles (id, full_name, phone)
  VALUES (
    NEW.id,
    COALESCE(NEW.raw_user_meta_data->>'full_name', ''),
    COALESCE(NEW.raw_user_meta_data->>'phone', '')
  );
  
  -- Assign default user role
  INSERT INTO public.user_roles (user_id, role)
  VALUES (NEW.id, 'user');
  
  RETURN NEW;
END;
$$;

-- Trigger for new user signup
CREATE TRIGGER on_auth_user_created
  AFTER INSERT ON auth.users
  FOR EACH ROW EXECUTE FUNCTION public.handle_new_user();

-- Function to update updated_at timestamp
CREATE OR REPLACE FUNCTION public.update_updated_at()
RETURNS TRIGGER
LANGUAGE plpgsql
AS $$
BEGIN
  NEW.updated_at = NOW();
  RETURN NEW;
END;
$$;

-- Trigger for profiles updated_at
CREATE TRIGGER update_profiles_updated_at
  BEFORE UPDATE ON public.profiles
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();

-- Insert default modules
INSERT INTO public.modules (name, name_bn, description, description_bn, icon, route, order_index) VALUES
  ('Smart Marketer', 'স্মার্ট মার্কেটার', 'Marketing automation and insights', 'মার্কেটিং অটোমেশন এবং অন্তর্দৃষ্টি', '📈', '/modules/marketer', 1),
  ('Smart Design Hub', 'স্মার্ট ডিজাইন হাব', 'Design tools and templates', 'ডিজাইন টুলস এবং টেমপ্লেট', '🎨', '/modules/design', 2),
  ('Smart Academy', 'স্মার্ট একাডেমি', 'Online learning and exams', 'অনলাইন শিক্ষা এবং পরীক্ষা', '📚', '/modules/academy', 3),
  ('Smart Hospital', 'স্মার্ট হাসপাতাল', 'Healthcare and doctor consultation', 'স্বাস্থ্যসেবা এবং ডাক্তার পরামর্শ', '🏥', '/modules/hospital', 4),
  ('Smart News', 'স্মার্ট নিউজ', 'Latest news and updates', 'সর্বশেষ সংবাদ এবং আপডেট', '📰', '/modules/news', 5),
  ('Smart Pay', 'স্মার্ট পে', 'Payment and transactions', 'পেমেন্ট এবং লেনদেন', '💳', '/modules/pay', 6),
  ('Smart FNF', 'স্মার্ট এফএনএফ', 'Social networking platform', 'সামাজিক নেটওয়ার্কিং প্ল্যাটফর্ম', '👥', '/modules/fnf', 7),
  ('Smart Tourist', 'স্মার্ট ট্যুরিস্ট', 'Travel and tourism guide', 'ভ্রমণ এবং পর্যটন গাইড', '✈️', '/modules/tourist', 8),
  ('Smart Bazar', 'স্মার্ট বাজার', 'Online marketplace', 'অনলাইন মার্কেটপ্লেস', '🛒', '/modules/bazar', 9),
  ('Smart Lawyer', 'স্মার্ট আইনজীবী', 'Legal consultation and advice', 'আইনি পরামর্শ এবং উপদেশ', '⚖️', '/modules/lawyer', 10),
  ('Smart Hisab', 'স্মার্ট হিসাব', 'Accounting and finance', 'হিসাব এবং অর্থ', '💰', '/modules/hisab', 11),
  ('Auto Lottery', 'অটো লটারি', 'Lottery system', 'লটারি সিস্টেম', '🎰', '/modules/lottery', 12),
  ('Smart Desk', 'স্মার্ট ডেস্ক', 'Remote desktop access', 'রিমোট ডেস্কটপ অ্যাক্সেস', '🖥️', '/modules/desk', 13);