-- Create core_brain_logs table
CREATE TABLE public.core_brain_logs (
  id uuid NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  type text NOT NULL,
  message text NOT NULL,
  details jsonb,
  created_at timestamp with time zone NOT NULL DEFAULT now(),
  user_id uuid REFERENCES auth.users(id) ON DELETE SET NULL
);

-- Create core_brain_plans table
CREATE TABLE public.core_brain_plans (
  id uuid NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  title text NOT NULL,
  description text NOT NULL,
  status text NOT NULL DEFAULT 'pending',
  created_at timestamp with time zone NOT NULL DEFAULT now(),
  updated_at timestamp with time zone NOT NULL DEFAULT now(),
  created_by uuid NOT NULL REFERENCES auth.users(id) ON DELETE CASCADE
);

-- Enable RLS
ALTER TABLE public.core_brain_logs ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.core_brain_plans ENABLE ROW LEVEL SECURITY;

-- Create function to check if user is a core brain owner
CREATE OR REPLACE FUNCTION public.is_core_brain_owner()
RETURNS boolean
LANGUAGE sql
STABLE
SECURITY DEFINER
SET search_path = public
AS $$
  SELECT EXISTS (
    SELECT 1
    FROM auth.users
    WHERE id = auth.uid()
      AND email IN (
        'nazrul.tasnim24@gmail.com',
        'engineer.tasnim@smartseba.com'
      )
  )
$$;

-- RLS Policies for core_brain_logs
CREATE POLICY "Core Brain owners can view logs"
ON public.core_brain_logs
FOR SELECT
USING (public.is_core_brain_owner());

CREATE POLICY "Core Brain owners can insert logs"
ON public.core_brain_logs
FOR INSERT
WITH CHECK (public.is_core_brain_owner());

-- RLS Policies for core_brain_plans
CREATE POLICY "Core Brain owners can view plans"
ON public.core_brain_plans
FOR SELECT
USING (public.is_core_brain_owner());

CREATE POLICY "Core Brain owners can insert plans"
ON public.core_brain_plans
FOR INSERT
WITH CHECK (public.is_core_brain_owner() AND auth.uid() = created_by);

CREATE POLICY "Core Brain owners can update plans"
ON public.core_brain_plans
FOR UPDATE
USING (public.is_core_brain_owner());

CREATE POLICY "Core Brain owners can delete plans"
ON public.core_brain_plans
FOR DELETE
USING (public.is_core_brain_owner());

-- Create trigger for updated_at
CREATE TRIGGER update_core_brain_plans_updated_at
BEFORE UPDATE ON public.core_brain_plans
FOR EACH ROW
EXECUTE FUNCTION public.update_updated_at();

-- Insert some sample logs for testing
INSERT INTO public.core_brain_logs (type, message, details)
VALUES 
  ('error', 'Connection timeout in chat module', '{"module": "chat", "duration": "5000ms"}'::jsonb),
  ('warning', 'Rate limit exceeded for AI API', '{"api": "lovable-ai", "limit": "100/min"}'::jsonb),
  ('warning', 'WebRTC connection failed for user', '{"protocol": "webrtc", "error": "ICE failed"}'::jsonb),
  ('info', 'Database query slow', '{"query": "SELECT * FROM messages", "duration": "2100ms"}'::jsonb);