-- Create scan_results table for storing project scan findings
CREATE TABLE public.scan_results (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  scan_type TEXT NOT NULL, -- 'frontend', 'backend', 'database', 'full'
  file_path TEXT,
  issue_type TEXT NOT NULL, -- 'error', 'warning', 'unused', 'outdated', 'missing', 'performance'
  severity TEXT NOT NULL DEFAULT 'medium', -- 'low', 'medium', 'high', 'critical'
  message TEXT NOT NULL,
  details JSONB,
  resolved BOOLEAN DEFAULT false,
  created_at TIMESTAMPTZ NOT NULL DEFAULT now()
);

-- Create fix_plans table for storing AI-generated fix plans
CREATE TABLE public.fix_plans (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  scan_result_id UUID REFERENCES public.scan_results(id) ON DELETE SET NULL,
  title TEXT NOT NULL,
  description TEXT NOT NULL,
  priority TEXT NOT NULL DEFAULT 'medium', -- 'low', 'medium', 'high', 'critical'
  status TEXT NOT NULL DEFAULT 'pending', -- 'pending', 'approved', 'rejected', 'completed'
  ai_suggestion TEXT,
  created_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  updated_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  created_by UUID NOT NULL
);

-- Create code_patches table for storing generated code patches
CREATE TABLE public.code_patches (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  fix_plan_id UUID REFERENCES public.fix_plans(id) ON DELETE CASCADE,
  file_path TEXT NOT NULL,
  original_code TEXT,
  patched_code TEXT NOT NULL,
  status TEXT NOT NULL DEFAULT 'pending', -- 'pending', 'approved', 'rejected', 'applied'
  applied_at TIMESTAMPTZ,
  created_at TIMESTAMPTZ NOT NULL DEFAULT now(),
  created_by UUID NOT NULL
);

-- Enable RLS on all tables
ALTER TABLE public.scan_results ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.fix_plans ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.code_patches ENABLE ROW LEVEL SECURITY;

-- RLS Policies for scan_results
CREATE POLICY "Core Brain owners can view scan results"
  ON public.scan_results FOR SELECT
  USING (is_core_brain_owner());

CREATE POLICY "Core Brain owners can insert scan results"
  ON public.scan_results FOR INSERT
  WITH CHECK (is_core_brain_owner());

CREATE POLICY "Core Brain owners can update scan results"
  ON public.scan_results FOR UPDATE
  USING (is_core_brain_owner());

CREATE POLICY "Core Brain owners can delete scan results"
  ON public.scan_results FOR DELETE
  USING (is_core_brain_owner());

-- RLS Policies for fix_plans
CREATE POLICY "Core Brain owners can view fix plans"
  ON public.fix_plans FOR SELECT
  USING (is_core_brain_owner());

CREATE POLICY "Core Brain owners can insert fix plans"
  ON public.fix_plans FOR INSERT
  WITH CHECK (is_core_brain_owner() AND auth.uid() = created_by);

CREATE POLICY "Core Brain owners can update fix plans"
  ON public.fix_plans FOR UPDATE
  USING (is_core_brain_owner());

CREATE POLICY "Core Brain owners can delete fix plans"
  ON public.fix_plans FOR DELETE
  USING (is_core_brain_owner());

-- RLS Policies for code_patches
CREATE POLICY "Core Brain owners can view code patches"
  ON public.code_patches FOR SELECT
  USING (is_core_brain_owner());

CREATE POLICY "Core Brain owners can insert code patches"
  ON public.code_patches FOR INSERT
  WITH CHECK (is_core_brain_owner() AND auth.uid() = created_by);

CREATE POLICY "Core Brain owners can update code patches"
  ON public.code_patches FOR UPDATE
  USING (is_core_brain_owner());

CREATE POLICY "Core Brain owners can delete code patches"
  ON public.code_patches FOR DELETE
  USING (is_core_brain_owner());

-- Add trigger for fix_plans updated_at
CREATE TRIGGER update_fix_plans_updated_at
  BEFORE UPDATE ON public.fix_plans
  FOR EACH ROW
  EXECUTE FUNCTION public.update_updated_at();