-- Create payment_requests table
CREATE TABLE public.payment_requests (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  requester_id UUID NOT NULL REFERENCES public.profiles(id) ON DELETE CASCADE,
  recipient_id UUID NOT NULL REFERENCES public.profiles(id) ON DELETE CASCADE,
  amount NUMERIC NOT NULL CHECK (amount > 0),
  note TEXT,
  status TEXT NOT NULL DEFAULT 'pending' CHECK (status IN ('pending', 'accepted', 'rejected', 'cancelled')),
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

-- Enable RLS
ALTER TABLE public.payment_requests ENABLE ROW LEVEL SECURITY;

-- Users can view payment requests where they are requester or recipient
CREATE POLICY "Users can view their payment requests"
ON public.payment_requests
FOR SELECT
USING (auth.uid() = requester_id OR auth.uid() = recipient_id);

-- Users can create payment requests where they are the requester
CREATE POLICY "Users can create payment requests"
ON public.payment_requests
FOR INSERT
WITH CHECK (auth.uid() = requester_id);

-- Users can update their own payment requests (cancel) or requests to them (accept/reject)
CREATE POLICY "Users can update payment requests"
ON public.payment_requests
FOR UPDATE
USING (
  (auth.uid() = requester_id AND status = 'pending') OR 
  (auth.uid() = recipient_id AND status = 'pending')
);

-- Trigger for updated_at
CREATE TRIGGER update_payment_requests_updated_at
BEFORE UPDATE ON public.payment_requests
FOR EACH ROW
EXECUTE FUNCTION public.update_updated_at();

-- Index for faster queries
CREATE INDEX idx_payment_requests_requester ON public.payment_requests(requester_id);
CREATE INDEX idx_payment_requests_recipient ON public.payment_requests(recipient_id);
CREATE INDEX idx_payment_requests_status ON public.payment_requests(status);

-- Enable realtime for payment requests
ALTER PUBLICATION supabase_realtime ADD TABLE public.payment_requests;