-- Enable pg_cron extension for scheduled jobs
CREATE EXTENSION IF NOT EXISTS pg_cron;

-- Grant necessary permissions to postgres role
GRANT USAGE ON SCHEMA cron TO postgres;
GRANT ALL PRIVILEGES ON ALL TABLES IN SCHEMA cron TO postgres;

-- Create a function to trigger the scanner via edge function
CREATE OR REPLACE FUNCTION public.trigger_core_brain_scan()
RETURNS void
LANGUAGE plpgsql
SECURITY DEFINER
SET search_path = public
AS $$
BEGIN
  -- Log the scheduled scan trigger
  INSERT INTO public.core_brain_logs (type, message, details)
  VALUES (
    'info',
    'Scheduled scan triggered',
    jsonb_build_object(
      'scan_type', 'scheduled',
      'triggered_at', now()
    )
  );
  
  -- Note: The actual scan will be performed by the edge function
  -- This just logs that a scan should happen
  PERFORM pg_notify('core_brain_scan', 'scheduled');
END;
$$;

-- Schedule the scan to run every 6 hours
-- This will run at 00:00, 06:00, 12:00, and 18:00 UTC daily
SELECT cron.schedule(
  'core-brain-scheduled-scan',
  '0 */6 * * *',
  $$SELECT public.trigger_core_brain_scan()$$
);

-- Create a table to track critical error notifications
CREATE TABLE IF NOT EXISTS public.critical_error_notifications (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  error_id UUID REFERENCES public.scan_results(id) ON DELETE CASCADE,
  notified_at TIMESTAMP WITH TIME ZONE DEFAULT now(),
  notification_type TEXT NOT NULL DEFAULT 'in-app',
  details JSONB,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

-- Enable RLS on critical_error_notifications
ALTER TABLE public.critical_error_notifications ENABLE ROW LEVEL SECURITY;

-- Allow Core Brain owners to view notifications
CREATE POLICY "Core Brain owners can view notifications"
  ON public.critical_error_notifications
  FOR SELECT
  USING (is_core_brain_owner());

-- Allow Core Brain owners to insert notifications
CREATE POLICY "Core Brain owners can insert notifications"
  ON public.critical_error_notifications
  FOR INSERT
  WITH CHECK (is_core_brain_owner());

-- Create index for faster queries
CREATE INDEX IF NOT EXISTS idx_critical_notifications_created 
  ON public.critical_error_notifications(created_at DESC);

-- Create index for error_id lookups
CREATE INDEX IF NOT EXISTS idx_critical_notifications_error 
  ON public.critical_error_notifications(error_id);