-- Create code_modules table
CREATE TABLE IF NOT EXISTS public.code_modules (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  created_by UUID NOT NULL REFERENCES auth.users(id) ON DELETE CASCADE,
  name TEXT NOT NULL,
  slug TEXT NOT NULL,
  description TEXT,
  category TEXT NOT NULL DEFAULT 'Core',
  module_config JSONB DEFAULT '{"permissions": [], "actions": [], "ai_suggestions": []}'::jsonb,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT now(),
  UNIQUE(created_by, slug)
);

-- Create smart_wallets table
CREATE TABLE IF NOT EXISTS public.smart_wallets (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id) ON DELETE CASCADE UNIQUE,
  balance NUMERIC(10, 2) NOT NULL DEFAULT 0.00,
  currency TEXT DEFAULT 'BDT',
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

-- Create smart_subscriptions table
CREATE TABLE IF NOT EXISTS public.smart_subscriptions (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id) ON DELETE CASCADE UNIQUE,
  plan TEXT NOT NULL DEFAULT 'Basic',
  expires_at TIMESTAMP WITH TIME ZONE NOT NULL,
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

-- Create wallet_transactions table for transaction history
CREATE TABLE IF NOT EXISTS public.wallet_transactions (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id) ON DELETE CASCADE,
  amount NUMERIC(10, 2) NOT NULL,
  type TEXT NOT NULL,
  description TEXT,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

-- Enable RLS
ALTER TABLE public.code_modules ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.smart_wallets ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.smart_subscriptions ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.wallet_transactions ENABLE ROW LEVEL SECURITY;

-- RLS Policies for code_modules
CREATE POLICY "Users can view their own modules" ON public.code_modules
  FOR SELECT USING (auth.uid() = created_by);

CREATE POLICY "Users can create their own modules" ON public.code_modules
  FOR INSERT WITH CHECK (auth.uid() = created_by);

CREATE POLICY "Users can update their own modules" ON public.code_modules
  FOR UPDATE USING (auth.uid() = created_by);

CREATE POLICY "Users can delete their own modules" ON public.code_modules
  FOR DELETE USING (auth.uid() = created_by);

-- RLS Policies for smart_wallets
CREATE POLICY "Users can view their own wallet" ON public.smart_wallets
  FOR SELECT USING (auth.uid() = user_id);

CREATE POLICY "Users can insert their own wallet" ON public.smart_wallets
  FOR INSERT WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update their own wallet" ON public.smart_wallets
  FOR UPDATE USING (auth.uid() = user_id);

-- RLS Policies for smart_subscriptions
CREATE POLICY "Users can view their own subscription" ON public.smart_subscriptions
  FOR SELECT USING (auth.uid() = user_id);

CREATE POLICY "Users can insert their own subscription" ON public.smart_subscriptions
  FOR INSERT WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update their own subscription" ON public.smart_subscriptions
  FOR UPDATE USING (auth.uid() = user_id);

-- RLS Policies for wallet_transactions
CREATE POLICY "Users can view their own transactions" ON public.wallet_transactions
  FOR SELECT USING (auth.uid() = user_id);

CREATE POLICY "Users can insert their own transactions" ON public.wallet_transactions
  FOR INSERT WITH CHECK (auth.uid() = user_id);

-- Triggers for updated_at
CREATE TRIGGER update_code_modules_updated_at
  BEFORE UPDATE ON public.code_modules
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();

CREATE TRIGGER update_smart_wallets_updated_at
  BEFORE UPDATE ON public.smart_wallets
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();

CREATE TRIGGER update_smart_subscriptions_updated_at
  BEFORE UPDATE ON public.smart_subscriptions
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();