-- Create project_files table for IDE file system
CREATE TABLE IF NOT EXISTS public.project_files (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL,
  project_id UUID NOT NULL DEFAULT gen_random_uuid(),
  file_path TEXT NOT NULL,
  file_name TEXT NOT NULL,
  file_type TEXT NOT NULL DEFAULT 'file',
  content TEXT,
  parent_id UUID,
  created_at TIMESTAMPTZ DEFAULT now(),
  updated_at TIMESTAMPTZ DEFAULT now(),
  UNIQUE(user_id, project_id, file_path)
);

-- Create ai_changes table for change history
CREATE TABLE IF NOT EXISTS public.ai_changes (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL,
  project_id UUID NOT NULL,
  file_path TEXT NOT NULL,
  original_code TEXT,
  generated_code TEXT NOT NULL,
  prompt TEXT NOT NULL,
  applied BOOLEAN DEFAULT false,
  created_at TIMESTAMPTZ DEFAULT now()
);

-- Create user_memory table for long-term memory
CREATE TABLE IF NOT EXISTS public.user_memory (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL,
  memory_type TEXT NOT NULL,
  memory_key TEXT NOT NULL,
  memory_value JSONB NOT NULL,
  project_id UUID,
  created_at TIMESTAMPTZ DEFAULT now(),
  updated_at TIMESTAMPTZ DEFAULT now(),
  UNIQUE(user_id, memory_type, memory_key, project_id)
);

-- Create module_templates table for module definitions
CREATE TABLE IF NOT EXISTS public.module_templates (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  slug TEXT NOT NULL UNIQUE,
  description TEXT,
  category TEXT NOT NULL,
  template_config JSONB NOT NULL,
  is_multi_tenant BOOLEAN DEFAULT false,
  created_at TIMESTAMPTZ DEFAULT now(),
  updated_at TIMESTAMPTZ DEFAULT now()
);

-- Enable RLS
ALTER TABLE public.project_files ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.ai_changes ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.user_memory ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.module_templates ENABLE ROW LEVEL SECURITY;

-- RLS Policies for project_files
CREATE POLICY "Users can view their own files" ON public.project_files
  FOR SELECT USING (auth.uid() = user_id);

CREATE POLICY "Users can insert their own files" ON public.project_files
  FOR INSERT WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update their own files" ON public.project_files
  FOR UPDATE USING (auth.uid() = user_id);

CREATE POLICY "Users can delete their own files" ON public.project_files
  FOR DELETE USING (auth.uid() = user_id);

-- RLS Policies for ai_changes
CREATE POLICY "Users can view their own changes" ON public.ai_changes
  FOR SELECT USING (auth.uid() = user_id);

CREATE POLICY "Users can insert their own changes" ON public.ai_changes
  FOR INSERT WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update their own changes" ON public.ai_changes
  FOR UPDATE USING (auth.uid() = user_id);

-- RLS Policies for user_memory
CREATE POLICY "Users can view their own memory" ON public.user_memory
  FOR SELECT USING (auth.uid() = user_id);

CREATE POLICY "Users can insert their own memory" ON public.user_memory
  FOR INSERT WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update their own memory" ON public.user_memory
  FOR UPDATE USING (auth.uid() = user_id);

CREATE POLICY "Users can delete their own memory" ON public.user_memory
  FOR DELETE USING (auth.uid() = user_id);

-- RLS Policies for module_templates
CREATE POLICY "Anyone can view module templates" ON public.module_templates
  FOR SELECT USING (true);

CREATE POLICY "Admins can manage templates" ON public.module_templates
  FOR ALL USING (has_role(auth.uid(), 'admin'::app_role));

-- Triggers for updated_at
CREATE TRIGGER update_project_files_updated_at
  BEFORE UPDATE ON public.project_files
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();

CREATE TRIGGER update_user_memory_updated_at
  BEFORE UPDATE ON public.user_memory
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();

CREATE TRIGGER update_module_templates_updated_at
  BEFORE UPDATE ON public.module_templates
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();

-- Insert default module templates
INSERT INTO public.module_templates (name, slug, description, category, template_config, is_multi_tenant) VALUES
('Smart Salon', 'smart-salon', 'Complete salon management system', 'Business', '{"framework": "laravel", "features": ["booking", "inventory", "staff", "billing"]}', true),
('Smart Makeup', 'smart-makeup', 'Makeup artist booking platform', 'Beauty', '{"framework": "laravel", "features": ["booking", "portfolio", "reviews"]}', true),
('Smart Bazar', 'smart-bazar', 'Multi-vendor e-commerce platform', 'E-commerce', '{"framework": "laravel", "features": ["multi-vendor", "payment", "shipping", "inventory"]}', true),
('Smart Academy', 'smart-academy', 'Online learning management system', 'Education', '{"framework": "laravel", "features": ["courses", "students", "exams", "certificates"]}', true),
('Smart Hospital', 'smart-hospital', 'Hospital management system', 'Healthcare', '{"framework": "laravel", "features": ["appointments", "patients", "billing", "pharmacy"]}', true),
('Smart Doctor', 'smart-doctor', 'Doctor appointment system', 'Healthcare', '{"framework": "laravel", "features": ["appointments", "prescriptions", "telemedicine"]}', false),
('Smart Lawyer', 'smart-lawyer', 'Legal case management', 'Legal', '{"framework": "laravel", "features": ["cases", "clients", "documents", "billing"]}', false),
('Smart Pay', 'smart-pay', 'Payment gateway integration', 'FinTech', '{"framework": "laravel", "features": ["bkash", "nagad", "card", "wallet"]}', false),
('Smart ERP', 'smart-erp', 'Enterprise resource planning', 'Business', '{"framework": "laravel", "features": ["hr", "inventory", "accounting", "crm"]}', true),
('Smart News', 'smart-news', 'News publishing platform', 'Media', '{"framework": "laravel", "features": ["articles", "categories", "authors", "comments"]}', true)
ON CONFLICT (slug) DO NOTHING;