-- Affiliate System Tables
CREATE TABLE IF NOT EXISTS public.affiliate_links (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL,
  referral_code TEXT UNIQUE NOT NULL,
  module_slug TEXT,
  clicks INTEGER DEFAULT 0,
  conversions INTEGER DEFAULT 0,
  total_earnings DECIMAL(10,2) DEFAULT 0,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.affiliate_conversions (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  affiliate_link_id UUID REFERENCES public.affiliate_links(id) ON DELETE CASCADE,
  referred_user_id UUID,
  module_slug TEXT,
  commission_amount DECIMAL(10,2),
  status TEXT DEFAULT 'pending',
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.affiliate_payouts (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL,
  amount DECIMAL(10,2) NOT NULL,
  status TEXT DEFAULT 'pending',
  payment_method TEXT,
  payment_details JSONB,
  processed_at TIMESTAMP WITH TIME ZONE,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.affiliate_settings (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  module_slug TEXT UNIQUE NOT NULL,
  commission_rate DECIMAL(5,2) DEFAULT 10.00,
  enabled BOOLEAN DEFAULT true,
  min_payout DECIMAL(10,2) DEFAULT 50.00,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

-- Enable RLS
ALTER TABLE public.affiliate_links ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.affiliate_conversions ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.affiliate_payouts ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.affiliate_settings ENABLE ROW LEVEL SECURITY;

-- RLS Policies for affiliate_links
CREATE POLICY "Users can view own affiliate links"
  ON public.affiliate_links FOR SELECT
  USING (auth.uid() = user_id);

CREATE POLICY "Users can create own affiliate links"
  ON public.affiliate_links FOR INSERT
  WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update own affiliate links"
  ON public.affiliate_links FOR UPDATE
  USING (auth.uid() = user_id);

-- RLS Policies for affiliate_conversions
CREATE POLICY "Users can view own conversions"
  ON public.affiliate_conversions FOR SELECT
  USING (
    EXISTS (
      SELECT 1 FROM public.affiliate_links
      WHERE id = affiliate_link_id AND user_id = auth.uid()
    )
  );

CREATE POLICY "Admins can manage conversions"
  ON public.affiliate_conversions FOR ALL
  USING (public.has_role(auth.uid(), 'admin'));

-- RLS Policies for affiliate_payouts
CREATE POLICY "Users can view own payouts"
  ON public.affiliate_payouts FOR SELECT
  USING (auth.uid() = user_id);

CREATE POLICY "Admins can manage payouts"
  ON public.affiliate_payouts FOR ALL
  USING (public.has_role(auth.uid(), 'admin'));

-- RLS Policies for affiliate_settings
CREATE POLICY "Everyone can view affiliate settings"
  ON public.affiliate_settings FOR SELECT
  USING (true);

CREATE POLICY "Admins can manage affiliate settings"
  ON public.affiliate_settings FOR ALL
  USING (public.has_role(auth.uid(), 'admin'));

-- Triggers for updated_at
CREATE OR REPLACE FUNCTION public.update_affiliate_updated_at()
RETURNS TRIGGER AS $$
BEGIN
  NEW.updated_at = now();
  RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER update_affiliate_links_updated_at
  BEFORE UPDATE ON public.affiliate_links
  FOR EACH ROW
  EXECUTE FUNCTION public.update_affiliate_updated_at();

CREATE TRIGGER update_affiliate_settings_updated_at
  BEFORE UPDATE ON public.affiliate_settings
  FOR EACH ROW
  EXECUTE FUNCTION public.update_affiliate_updated_at();