-- PHASE-3: Smart Seba Module System
-- Creating tables for all 14 priority modules

-- ===============================================
-- MODULE 1: SMART MARKETER
-- ===============================================
CREATE TABLE IF NOT EXISTS public.marketing_campaigns (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id),
  title TEXT NOT NULL,
  content TEXT,
  platform TEXT, -- facebook, google, youtube
  status TEXT DEFAULT 'draft',
  scheduled_at TIMESTAMPTZ,
  analytics JSONB DEFAULT '{}',
  created_at TIMESTAMPTZ DEFAULT now(),
  updated_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.marketing_campaigns ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Users can manage own campaigns" ON public.marketing_campaigns
  FOR ALL USING (auth.uid() = user_id);

-- ===============================================
-- MODULE 2: SMART DESIGN HUB
-- ===============================================
CREATE TABLE IF NOT EXISTS public.design_projects (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id),
  title TEXT NOT NULL,
  design_data JSONB DEFAULT '{}',
  template_id UUID,
  category TEXT, -- poster, banner, id_card
  export_format TEXT, -- png, pdf
  created_at TIMESTAMPTZ DEFAULT now(),
  updated_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.design_projects ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Users can manage own designs" ON public.design_projects
  FOR ALL USING (auth.uid() = user_id);

CREATE TABLE IF NOT EXISTS public.design_templates (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title TEXT NOT NULL,
  category TEXT NOT NULL,
  template_data JSONB DEFAULT '{}',
  thumbnail_url TEXT,
  is_premium BOOLEAN DEFAULT false,
  created_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.design_templates ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Anyone can view templates" ON public.design_templates
  FOR SELECT USING (true);

-- ===============================================
-- MODULE 3: SMART ACADEMY
-- ===============================================
CREATE TABLE IF NOT EXISTS public.academy_classes (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title TEXT NOT NULL,
  description TEXT,
  grade_level TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.academy_subjects (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  class_id UUID REFERENCES public.academy_classes(id),
  title TEXT NOT NULL,
  description TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.academy_chapters (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  subject_id UUID REFERENCES public.academy_subjects(id),
  title TEXT NOT NULL,
  content TEXT,
  video_url TEXT,
  order_index INTEGER DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.academy_exams (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  subject_id UUID REFERENCES public.academy_subjects(id),
  title TEXT NOT NULL,
  exam_type TEXT, -- mcq, cq
  questions JSONB DEFAULT '[]',
  duration_minutes INTEGER,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.academy_student_results (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id),
  exam_id UUID REFERENCES public.academy_exams(id),
  answers JSONB DEFAULT '{}',
  score NUMERIC,
  ai_evaluation TEXT,
  submitted_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.academy_classes ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.academy_subjects ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.academy_chapters ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.academy_exams ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.academy_student_results ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Anyone can view academy content" ON public.academy_classes FOR SELECT USING (true);
CREATE POLICY "Anyone can view subjects" ON public.academy_subjects FOR SELECT USING (true);
CREATE POLICY "Anyone can view chapters" ON public.academy_chapters FOR SELECT USING (true);
CREATE POLICY "Anyone can view exams" ON public.academy_exams FOR SELECT USING (true);
CREATE POLICY "Users can manage own results" ON public.academy_student_results FOR ALL USING (auth.uid() = user_id);

-- ===============================================
-- MODULE 5: SMART DOCTOR & HOSPITAL
-- ===============================================
CREATE TABLE IF NOT EXISTS public.hospital_departments (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  description TEXT,
  icon TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.doctors (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id),
  name TEXT NOT NULL,
  specialty TEXT,
  department_id UUID REFERENCES public.hospital_departments(id),
  qualifications TEXT,
  experience_years INTEGER,
  consultation_fee NUMERIC,
  available_days TEXT[],
  avatar_url TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.appointments (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  patient_id UUID NOT NULL REFERENCES auth.users(id),
  doctor_id UUID REFERENCES public.doctors(id),
  appointment_date TIMESTAMPTZ NOT NULL,
  symptoms TEXT,
  status TEXT DEFAULT 'pending',
  prescription TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.patient_history (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  patient_id UUID NOT NULL REFERENCES auth.users(id),
  doctor_id UUID REFERENCES public.doctors(id),
  diagnosis TEXT,
  prescription TEXT,
  visit_date TIMESTAMPTZ DEFAULT now(),
  created_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.hospital_departments ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.doctors ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.appointments ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.patient_history ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Anyone can view departments" ON public.hospital_departments FOR SELECT USING (true);
CREATE POLICY "Anyone can view doctors" ON public.doctors FOR SELECT USING (true);
CREATE POLICY "Users can manage own appointments" ON public.appointments FOR ALL USING (auth.uid() = patient_id);
CREATE POLICY "Users can view own history" ON public.patient_history FOR SELECT USING (auth.uid() = patient_id);

-- ===============================================
-- MODULE 6: SMART NEWS
-- ===============================================
CREATE TABLE IF NOT EXISTS public.news_articles (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  author_id UUID REFERENCES auth.users(id),
  title TEXT NOT NULL,
  content TEXT NOT NULL,
  summary TEXT,
  category TEXT,
  thumbnail_url TEXT,
  is_trending BOOLEAN DEFAULT false,
  views INTEGER DEFAULT 0,
  published_at TIMESTAMPTZ DEFAULT now(),
  created_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.news_articles ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Anyone can view published news" ON public.news_articles FOR SELECT USING (true);
CREATE POLICY "Authors can manage own articles" ON public.news_articles FOR ALL USING (auth.uid() = author_id);

-- ===============================================
-- MODULE 8: SMART BAZAR (Multi-Tenant)
-- ===============================================
CREATE TABLE IF NOT EXISTS public.bazar_merchants (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id),
  business_name TEXT NOT NULL,
  business_logo TEXT,
  custom_domain TEXT,
  commission_rate NUMERIC DEFAULT 10,
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.bazar_products (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  merchant_id UUID REFERENCES public.bazar_merchants(id),
  title TEXT NOT NULL,
  description TEXT,
  price NUMERIC NOT NULL,
  stock INTEGER DEFAULT 0,
  images TEXT[],
  category TEXT,
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.bazar_orders (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  customer_id UUID NOT NULL REFERENCES auth.users(id),
  merchant_id UUID REFERENCES public.bazar_merchants(id),
  products JSONB DEFAULT '[]',
  total_amount NUMERIC NOT NULL,
  commission_amount NUMERIC,
  status TEXT DEFAULT 'pending',
  delivery_address TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.bazar_merchants ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.bazar_products ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.bazar_orders ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Anyone can view active merchants" ON public.bazar_merchants FOR SELECT USING (is_active = true);
CREATE POLICY "Merchants can manage own business" ON public.bazar_merchants FOR ALL USING (auth.uid() = user_id);
CREATE POLICY "Anyone can view active products" ON public.bazar_products FOR SELECT USING (is_active = true);
CREATE POLICY "Customers can view own orders" ON public.bazar_orders FOR SELECT USING (auth.uid() = customer_id);

-- ===============================================
-- MODULE 9: SMART TOURIST
-- ===============================================
CREATE TABLE IF NOT EXISTS public.tourist_places (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  description TEXT,
  location TEXT,
  category TEXT,
  images TEXT[],
  rating NUMERIC DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.tourist_hotels (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  location TEXT,
  price_per_night NUMERIC,
  amenities TEXT[],
  images TEXT[],
  rating NUMERIC DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.tour_packages (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title TEXT NOT NULL,
  description TEXT,
  destinations TEXT[],
  duration_days INTEGER,
  price NUMERIC,
  includes TEXT[],
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.tour_bookings (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id),
  package_id UUID REFERENCES public.tour_packages(id),
  booking_date TIMESTAMPTZ NOT NULL,
  travelers_count INTEGER,
  status TEXT DEFAULT 'pending',
  created_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.tourist_places ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.tourist_hotels ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.tour_packages ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.tour_bookings ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Anyone can view places" ON public.tourist_places FOR SELECT USING (true);
CREATE POLICY "Anyone can view hotels" ON public.tourist_hotels FOR SELECT USING (true);
CREATE POLICY "Anyone can view packages" ON public.tour_packages FOR SELECT USING (true);
CREATE POLICY "Users can manage own bookings" ON public.tour_bookings FOR ALL USING (auth.uid() = user_id);

-- ===============================================
-- MODULE 10: SMART JOB
-- ===============================================
CREATE TABLE IF NOT EXISTS public.job_postings (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  employer_id UUID NOT NULL REFERENCES auth.users(id),
  title TEXT NOT NULL,
  description TEXT,
  company_name TEXT,
  location TEXT,
  salary_range TEXT,
  job_type TEXT,
  requirements TEXT[],
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.job_applications (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  job_id UUID REFERENCES public.job_postings(id),
  applicant_id UUID NOT NULL REFERENCES auth.users(id),
  cv_url TEXT,
  cover_letter TEXT,
  status TEXT DEFAULT 'pending',
  ai_analysis TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.user_cvs (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id),
  cv_data JSONB DEFAULT '{}',
  cv_url TEXT,
  created_at TIMESTAMPTZ DEFAULT now(),
  updated_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.job_postings ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.job_applications ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.user_cvs ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Anyone can view active jobs" ON public.job_postings FOR SELECT USING (is_active = true);
CREATE POLICY "Employers can manage own jobs" ON public.job_postings FOR ALL USING (auth.uid() = employer_id);
CREATE POLICY "Users can manage own applications" ON public.job_applications FOR ALL USING (auth.uid() = applicant_id);
CREATE POLICY "Users can manage own CVs" ON public.user_cvs FOR ALL USING (auth.uid() = user_id);

-- ===============================================
-- MODULE 11: SMART AI CALL CENTER
-- ===============================================
CREATE TABLE IF NOT EXISTS public.call_tickets (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  caller_id UUID REFERENCES auth.users(id),
  subject TEXT NOT NULL,
  description TEXT,
  priority TEXT DEFAULT 'medium',
  status TEXT DEFAULT 'open',
  ai_response TEXT,
  created_at TIMESTAMPTZ DEFAULT now(),
  resolved_at TIMESTAMPTZ
);

CREATE TABLE IF NOT EXISTS public.call_history (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id),
  call_duration INTEGER,
  transcript TEXT,
  ai_summary TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.call_tickets ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.call_history ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Users can manage own tickets" ON public.call_tickets FOR ALL USING (auth.uid() = caller_id);
CREATE POLICY "Users can view own call history" ON public.call_history FOR SELECT USING (auth.uid() = user_id);

-- ===============================================
-- MODULE 12: SMART AI AD FIRM
-- ===============================================
CREATE TABLE IF NOT EXISTS public.ad_campaigns (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID NOT NULL REFERENCES auth.users(id),
  title TEXT NOT NULL,
  script TEXT,
  video_concept TEXT,
  storyboard JSONB DEFAULT '[]',
  voiceover_url TEXT,
  status TEXT DEFAULT 'draft',
  created_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.ad_campaigns ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Users can manage own ad campaigns" ON public.ad_campaigns FOR ALL USING (auth.uid() = user_id);

-- ===============================================
-- MODULE 13: SMART SCHOOL & COLLEGE ERP
-- ===============================================
CREATE TABLE IF NOT EXISTS public.erp_schools (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  address TEXT,
  contact TEXT,
  logo_url TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.erp_students (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id),
  school_id UUID REFERENCES public.erp_schools(id),
  student_id TEXT NOT NULL,
  class_name TEXT,
  section TEXT,
  roll_number TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.erp_attendance (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  student_id UUID REFERENCES public.erp_students(id),
  date DATE NOT NULL,
  status TEXT DEFAULT 'present',
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.erp_homework (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  student_id UUID REFERENCES public.erp_students(id),
  subject TEXT NOT NULL,
  title TEXT NOT NULL,
  description TEXT,
  due_date DATE,
  status TEXT DEFAULT 'pending',
  created_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.erp_schools ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.erp_students ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.erp_attendance ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.erp_homework ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Anyone can view schools" ON public.erp_schools FOR SELECT USING (true);
CREATE POLICY "Students can view own data" ON public.erp_students FOR SELECT USING (auth.uid() = user_id);
CREATE POLICY "Students can view own attendance" ON public.erp_attendance FOR SELECT USING (true);
CREATE POLICY "Students can manage own homework" ON public.erp_homework FOR ALL USING (true);

-- ===============================================
-- MODULE 14: SMART LAWYER
-- ===============================================
CREATE TABLE IF NOT EXISTS public.lawyers (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id),
  name TEXT NOT NULL,
  specialty TEXT,
  experience_years INTEGER,
  consultation_fee NUMERIC,
  avatar_url TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.legal_cases (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  client_id UUID NOT NULL REFERENCES auth.users(id),
  lawyer_id UUID REFERENCES public.lawyers(id),
  title TEXT NOT NULL,
  description TEXT,
  case_type TEXT,
  status TEXT DEFAULT 'open',
  documents TEXT[],
  created_at TIMESTAMPTZ DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.legal_documents (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title TEXT NOT NULL,
  template_content TEXT,
  category TEXT,
  created_at TIMESTAMPTZ DEFAULT now()
);

ALTER TABLE public.lawyers ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.legal_cases ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.legal_documents ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Anyone can view lawyers" ON public.lawyers FOR SELECT USING (true);
CREATE POLICY "Users can manage own cases" ON public.legal_cases FOR ALL USING (auth.uid() = client_id);
CREATE POLICY "Anyone can view legal documents" ON public.legal_documents FOR SELECT USING (true);

-- ===============================================
-- Add update triggers
-- ===============================================
CREATE TRIGGER update_marketing_campaigns_updated_at BEFORE UPDATE ON public.marketing_campaigns
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();

CREATE TRIGGER update_design_projects_updated_at BEFORE UPDATE ON public.design_projects
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();

CREATE TRIGGER update_user_cvs_updated_at BEFORE UPDATE ON public.user_cvs
  FOR EACH ROW EXECUTE FUNCTION public.update_updated_at();