-- Phase 6: Smart Friend Ultra Upgrade Tables

-- 1) Offline Brain Table (for offline mode responses)
CREATE TABLE IF NOT EXISTS public.offline_brain (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  question TEXT NOT NULL,
  answer TEXT NOT NULL,
  category TEXT DEFAULT 'general',
  language TEXT DEFAULT 'bn',
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

CREATE INDEX idx_offline_brain_question ON public.offline_brain(question);
CREATE INDEX idx_offline_brain_category ON public.offline_brain(category);
CREATE INDEX idx_offline_brain_language ON public.offline_brain(language);

-- Enable RLS
ALTER TABLE public.offline_brain ENABLE ROW LEVEL SECURITY;

-- Everyone can read offline brain
CREATE POLICY "Anyone can view offline brain responses"
  ON public.offline_brain FOR SELECT
  USING (true);

-- Only authenticated users can add to offline brain
CREATE POLICY "Authenticated users can insert offline brain"
  ON public.offline_brain FOR INSERT
  WITH CHECK (auth.uid() IS NOT NULL);

-- 2) Identity Rules Table (permanent identity responses)
CREATE TABLE IF NOT EXISTS public.identity_rules (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  question_pattern TEXT NOT NULL,
  fixed_answer TEXT NOT NULL,
  language TEXT DEFAULT 'bn',
  priority INTEGER DEFAULT 100,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

CREATE INDEX idx_identity_rules_pattern ON public.identity_rules(question_pattern);

-- Enable RLS
ALTER TABLE public.identity_rules ENABLE ROW LEVEL SECURITY;

-- Everyone can read identity rules
CREATE POLICY "Anyone can view identity rules"
  ON public.identity_rules FOR SELECT
  USING (true);

-- Only owners can manage identity rules
CREATE POLICY "Owners can manage identity rules"
  ON public.identity_rules FOR ALL
  USING (is_core_brain_owner());

-- 3) Safety Filter Table (safety rules and filters)
CREATE TABLE IF NOT EXISTS public.safety_filter (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  rule TEXT NOT NULL,
  category TEXT NOT NULL,
  severity TEXT DEFAULT 'medium',
  enabled BOOLEAN DEFAULT true,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT now()
);

CREATE INDEX idx_safety_filter_category ON public.safety_filter(category);
CREATE INDEX idx_safety_filter_enabled ON public.safety_filter(enabled);

-- Enable RLS
ALTER TABLE public.safety_filter ENABLE ROW LEVEL SECURITY;

-- Everyone can read safety filters
CREATE POLICY "Anyone can view safety filters"
  ON public.safety_filter FOR SELECT
  USING (enabled = true);

-- Only owners can manage safety filters
CREATE POLICY "Owners can manage safety filters"
  ON public.safety_filter FOR ALL
  USING (is_core_brain_owner());

-- 4) Extend user_memory table for hybrid offline/online support
ALTER TABLE public.user_memory 
  ADD COLUMN IF NOT EXISTS is_synced BOOLEAN DEFAULT false,
  ADD COLUMN IF NOT EXISTS sync_priority INTEGER DEFAULT 0,
  ADD COLUMN IF NOT EXISTS local_only BOOLEAN DEFAULT false;

CREATE INDEX IF NOT EXISTS idx_user_memory_synced ON public.user_memory(is_synced);
CREATE INDEX IF NOT EXISTS idx_user_memory_local ON public.user_memory(local_only);

-- Insert default identity rules (Bengali)
INSERT INTO public.identity_rules (question_pattern, fixed_answer, language, priority) VALUES
  ('তোমাকে কে তৈরি করেছে', 'আমাকে তৈরি করেছেন আমার নির্মাতা ও মালিক — Engineer S M Nazrul Islam Tasnim, Smart Seba-এর হেডকোয়ার্টার থেকে।', 'bn', 1000),
  ('তাসনিম কে', 'হ্যাঁ, তিনি আমার নির্মাতা ও মালিক।', 'bn', 900),
  ('নজরুলকে চিন', 'হ্যাঁ, তিনি আমার নির্মাতা ও মালিক।', 'bn', 900),
  ('Tasnim ke', 'হ্যাঁ, তিনি আমার নির্মাতা ও মালিক।', 'bn', 900),
  ('তাসনিম এর বাড়ি', 'আমি ব্যক্তিগত ঠিকানা প্রকাশ করতে পারি না। তিনি Smart Seba HQ থেকে আমাকে পরিচালনা করেন।', 'bn', 850),
  ('who created you', 'I was created by my creator and owner — Engineer S M Nazrul Islam Tasnim, from Smart Seba headquarters.', 'en', 1000),
  ('who is tasnim', 'Yes, he is my creator and owner.', 'en', 900),
  ('tasnim address', 'I cannot disclose personal addresses. He operates me from Smart Seba HQ.', 'en', 850);

-- Insert default safety filter rules
INSERT INTO public.safety_filter (rule, category, severity) VALUES
  ('No personal addresses or location details', 'privacy', 'high'),
  ('No OTP or verification codes', 'security', 'critical'),
  ('No bank account or financial details', 'security', 'critical'),
  ('No passwords or sensitive credentials', 'security', 'critical'),
  ('No dangerous or illegal content', 'safety', 'critical'),
  ('No adult or inappropriate content in Ava Child Mode', 'child_safety', 'critical'),
  ('No medical diagnosis or prescription', 'professional', 'high'),
  ('No legal advice as binding counsel', 'professional', 'high');

-- Insert sample offline brain responses (Bengali)
INSERT INTO public.offline_brain (question, answer, category, language) VALUES
  ('হ্যালো', 'হ্যালো! আমি Smart Friend। আপনাকে কিভাবে সাহায্য করতে পারি?', 'greeting', 'bn'),
  ('কেমন আছো', 'আমি ভালো আছি, ধন্যবাদ! আপনি কেমন আছেন?', 'greeting', 'bn'),
  ('তুমি কে', 'আমি Smart Friend, Smart Seba-এর AI সহায়ক। আমি আপনাকে বিভিন্ন বিষয়ে সাহায্য করতে পারি।', 'identity', 'bn'),
  ('ধন্যবাদ', 'আপনাকে স্বাগতম! আর কিছু জানতে চান?', 'courtesy', 'bn'),
  ('hello', 'Hello! I am Smart Friend. How can I help you?', 'greeting', 'en'),
  ('how are you', 'I am doing well, thank you! How are you?', 'greeting', 'en'),
  ('who are you', 'I am Smart Friend, the AI assistant for Smart Seba. I can help you with various topics.', 'identity', 'en'),
  ('thank you', 'You are welcome! Anything else you would like to know?', 'courtesy', 'en');